<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Response;
use App\Http\Controllers\HomeController;

/*
|--------------------------------------------------------------------------
| API Routes for Wilayah (Geographic Data)
|--------------------------------------------------------------------------
*/

Route::prefix('api')->group(function () {
    Route::get('/provinces', [App\Http\Controllers\Api\WilayahController::class, 'getProvinces']);
    Route::get('/regencies', [App\Http\Controllers\Api\WilayahController::class, 'getRegencies']);
    Route::get('/districts', [App\Http\Controllers\Api\WilayahController::class, 'getDistricts']);
    Route::get('/villages', [App\Http\Controllers\Api\WilayahController::class, 'getVillages']);
    Route::get('/postal-code', [App\Http\Controllers\Api\WilayahController::class, 'getPostalCode']);
    
    // Additional routes for kepala form
    Route::get('/wilayah/provinsi', [App\Http\Controllers\Api\WilayahController::class, 'getProvinces']);
    Route::get('/wilayah/kabupaten', [App\Http\Controllers\Api\WilayahController::class, 'getRegencies']);
    Route::get('/wilayah/kecamatan', [App\Http\Controllers\Api\WilayahController::class, 'getDistricts']);
    Route::get('/wilayah/kelurahan', [App\Http\Controllers\Api\WilayahController::class, 'getVillages']);
    Route::get('/wilayah/kode-pos', [App\Http\Controllers\Api\WilayahController::class, 'getPostalCode']);
});

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "web" middleware group. Make something great!
|
*/

// Public landing route
Route::get('/', function () {
    if (Auth::check()) {
        return redirect()->route('dashboard');
    }
    return view('landing');
})->name('landing');

// Fallback: serve files from storage/app/public when web server can't read the symlink
Route::get('/storage/{path}', function (string $path) {
    $path = ltrim($path, '/');
    if (!Storage::disk('public')->exists($path)) {
        abort(404);
    }
    $fullPath = Storage::disk('public')->path($path);
    $mimeType = @mime_content_type($fullPath) ?: 'application/octet-stream';
    return Response::file($fullPath, [
        'Content-Type' => $mimeType,
        'Cache-Control' => 'public, max-age=31536000',
    ]);
})->where('path', '.*');

// Authentication routes (manual implementation for Laravel 10)
Route::get('/login', [App\Http\Controllers\Auth\LoginController::class, 'showLoginForm'])->name('login');
Route::post('/login', [App\Http\Controllers\Auth\LoginController::class, 'login']);
Route::post('/logout', [App\Http\Controllers\Auth\LoginController::class, 'logout'])->name('logout');

// Public template and import routes for testing
Route::get('/template/users', [App\Http\Controllers\TemplateController::class, 'downloadUsersTemplate'])->name('public.users.template');
Route::get('/import/users', [App\Http\Controllers\TemplateController::class, 'showImportForm'])->name('public.users.import');
Route::post('/import/users', [App\Http\Controllers\TemplateController::class, 'importUsers'])->name('public.users.import.store');

// Protected routes
Route::middleware(['auth'])->group(function () {
    // Home redirect based on role
    Route::get('/home', [HomeController::class, 'index'])->name('home');
    
    // Dashboard route for all authenticated users
    Route::get('/dashboard', [HomeController::class, 'index'])->name('dashboard');
    
    // Admin routes
    Route::prefix('admin')->name('admin.')->middleware(['role:admin'])->group(function () {
        Route::get('/dashboard', function() {
            return view('admin.dashboard');
        })->name('dashboard');
        
        // User Management
        Route::resource('users', App\Http\Controllers\Admin\UserController::class);
        Route::patch('/users/{user}/toggle-status', [App\Http\Controllers\Admin\UserController::class, 'toggleStatus'])->name('users.toggle-status');
        Route::post('/users/bulk-action', [App\Http\Controllers\Admin\UserController::class, 'bulkAction'])->name('users.bulk-action');
        Route::get('/users/export', [App\Http\Controllers\Admin\UserController::class, 'export'])->name('users.export');
        Route::get('/users/import', [App\Http\Controllers\Admin\UserController::class, 'showImport'])->name('users.show-import');
        Route::post('/users/import', [App\Http\Controllers\Admin\UserController::class, 'import'])->name('users.import');
        Route::get('/users/template', [App\Http\Controllers\Admin\UserController::class, 'downloadTemplate'])->name('users.download-template');
        
        // Lembaga Management
        // Custom routes must be defined BEFORE resource to avoid being captured by {lembaga}
        Route::get('/lembaga/template', [App\Http\Controllers\Admin\LembagaImportController::class, 'template'])->name('lembaga.template');
        Route::post('/lembaga/import', [App\Http\Controllers\Admin\LembagaImportController::class, 'import'])->name('lembaga.import');
        Route::post('/lembaga/assign-kepala', [App\Http\Controllers\Admin\LembagaController::class, 'assignKepala'])->name('lembaga.assign-kepala');
        Route::resource('lembaga', App\Http\Controllers\Admin\LembagaController::class);
        
        // Instrumen Management
        Route::resource('instrumen', App\Http\Controllers\Admin\InstrumenController::class)->parameters(['instrumen' => 'instrumen']);
        Route::patch('/instrumen/{instrumen}/toggle-status', [App\Http\Controllers\Admin\InstrumenController::class, 'toggleStatus'])->name('instrumen.toggle-status');
        Route::get('/instrumen/{instrumen}/export-excel', [App\Http\Controllers\Admin\InstrumenController::class, 'exportExcel'])->name('instrumen.export-excel');
        
        // Komponen Penilaian Management
        Route::get('/instrumen/{instrumen}/komponen', [App\Http\Controllers\Admin\KomponenPenilaianController::class, 'index'])->name('komponen-penilaian.index');
        Route::get('/instrumen/{instrumen}/komponen/create', [App\Http\Controllers\Admin\KomponenPenilaianController::class, 'create'])->name('komponen-penilaian.create');
        Route::post('/instrumen/{instrumen}/komponen', [App\Http\Controllers\Admin\KomponenPenilaianController::class, 'store'])->name('komponen-penilaian.store');
        Route::get('/komponen/{komponen}/edit', [App\Http\Controllers\Admin\KomponenPenilaianController::class, 'edit'])->name('komponen-penilaian.edit');
        Route::put('/komponen/{komponen}', [App\Http\Controllers\Admin\KomponenPenilaianController::class, 'update'])->name('komponen-penilaian.update');
        Route::delete('/komponen/{komponen}', [App\Http\Controllers\Admin\KomponenPenilaianController::class, 'destroy'])->name('komponen-penilaian.destroy');
        
        // Sub Komponen Management
        Route::get('/komponen/{komponen}/sub-komponen', [App\Http\Controllers\Admin\SubKomponenController::class, 'index'])->name('sub-komponen.index');
        Route::get('/komponen/{komponen}/sub-komponen/create', [App\Http\Controllers\Admin\SubKomponenController::class, 'create'])->name('sub-komponen.create');
        Route::post('/komponen/{komponen}/sub-komponen', [App\Http\Controllers\Admin\SubKomponenController::class, 'store'])->name('sub-komponen.store');
        Route::get('/komponen/{komponen}/sub-komponen/{subKomponen}/show', [App\Http\Controllers\Admin\SubKomponenController::class, 'show'])->name('sub-komponen.show');
        Route::get('/komponen/{komponen}/sub-komponen/{subKomponen}/edit', [App\Http\Controllers\Admin\SubKomponenController::class, 'edit'])->name('sub-komponen.edit');
        Route::put('/komponen/{komponen}/sub-komponen/{subKomponen}', [App\Http\Controllers\Admin\SubKomponenController::class, 'update'])->name('sub-komponen.update');
        Route::delete('/komponen/{komponen}/sub-komponen/{subKomponen}', [App\Http\Controllers\Admin\SubKomponenController::class, 'destroy'])->name('sub-komponen.destroy');
        
        // Instrumen Penilaian Management (dengan sub komponen)
        Route::get('/komponen/{komponen}/sub-komponen/{subKomponen}/instrumen', [App\Http\Controllers\Admin\InstrumenPenilaianController::class, 'index'])->name('instrumen-penilaian.index');
        Route::get('/komponen/{komponen}/sub-komponen/{subKomponen}/instrumen/create', [App\Http\Controllers\Admin\InstrumenPenilaianController::class, 'create'])->name('instrumen-penilaian.create');
        Route::post('/komponen/{komponen}/sub-komponen/{subKomponen}/instrumen', [App\Http\Controllers\Admin\InstrumenPenilaianController::class, 'store'])->name('instrumen-penilaian.store');
        Route::get('/instrumen-penilaian/{instrumenPenilaian}/edit', [App\Http\Controllers\Admin\InstrumenPenilaianController::class, 'edit'])->name('instrumen-penilaian.edit');
        Route::put('/instrumen-penilaian/{instrumenPenilaian}', [App\Http\Controllers\Admin\InstrumenPenilaianController::class, 'update'])->name('instrumen-penilaian.update');
        Route::delete('/instrumen-penilaian/{instrumenPenilaian}', [App\Http\Controllers\Admin\InstrumenPenilaianController::class, 'destroy'])->name('instrumen-penilaian.destroy');
        Route::get('/instrumen-penilaian/{instrumenPenilaian}', [App\Http\Controllers\Admin\InstrumenPenilaianController::class, 'show'])->name('instrumen-penilaian.show');
        
        // Import/Export Instrumen Penilaian (dengan sub komponen)
        Route::get('/komponen/{komponen}/sub-komponen/{subKomponen}/instrumen/template', [App\Http\Controllers\Admin\InstrumenPenilaianImportController::class, 'template'])->name('instrumen-penilaian.template');
        Route::post('/komponen/{komponen}/sub-komponen/{subKomponen}/instrumen/import', [App\Http\Controllers\Admin\InstrumenPenilaianImportController::class, 'import'])->name('instrumen-penilaian.import');
        
        // Settings Management
        Route::get('/settings', [App\Http\Controllers\Admin\SettingsController::class, 'index'])->name('settings.index');
        Route::post('/settings/profile', [App\Http\Controllers\Admin\SettingsController::class, 'updateProfile'])->name('settings.update-profile');
        Route::post('/settings/password', [App\Http\Controllers\Admin\SettingsController::class, 'updatePassword'])->name('settings.update-password');
        Route::post('/settings/app', [App\Http\Controllers\Admin\SettingsController::class, 'updateAppSettings'])->name('settings.update-app');
        Route::post('/settings/system', [App\Http\Controllers\Admin\SettingsController::class, 'updateSystemSettings'])->name('settings.update-system');
        Route::post('/settings/gemini', [App\Http\Controllers\Admin\SettingsController::class, 'updateGeminiSettings'])->name('settings.update-gemini');
        Route::delete('/settings/logo', [App\Http\Controllers\Admin\SettingsController::class, 'deleteLogo'])->name('settings.delete-logo');
        Route::delete('/settings/favicon', [App\Http\Controllers\Admin\SettingsController::class, 'deleteFavicon'])->name('settings.delete-favicon');
    });
    
    // AJAX routes for pengawas (no middleware, validation in controller)
    Route::get('/pengawas/distribusi/komponen', [App\Http\Controllers\Pengawas\DistribusiController::class, 'getKomponen'])->name('pengawas.distribusi.get-komponen');
    
    // Pengawas routes
    Route::prefix('pengawas')->name('pengawas.')->middleware(['role:pengawas'])->group(function () {
        Route::get('/dashboard', function() {
            return view('pengawas.dashboard');
        })->name('dashboard');
        // Dokumentasi Pengawas
        Route::get('/dokumentasi', function () {
            return view('docs.guide', ['role' => 'pengawas']);
        })->name('docs');
        
        // Lembaga Management
        Route::resource('lembaga', App\Http\Controllers\Pengawas\LembagaController::class);
        Route::post('/lembaga/claim', [App\Http\Controllers\Pengawas\LembagaController::class, 'claim'])->name('lembaga.claim');
        
        // Instrumen Management (hanya index, show, edit, update - tidak ada create)
        Route::get('/instrumen', [App\Http\Controllers\Pengawas\InstrumenController::class, 'index'])->name('instrumen.index');
        Route::get('/instrumen/{instrumen}', [App\Http\Controllers\Pengawas\InstrumenController::class, 'show'])->name('instrumen.show');
        Route::get('/instrumen/{instrumen}/edit', [App\Http\Controllers\Pengawas\InstrumenController::class, 'edit'])->name('instrumen.edit');
        Route::put('/instrumen/{instrumen}', [App\Http\Controllers\Pengawas\InstrumenController::class, 'update'])->name('instrumen.update');
        Route::delete('/instrumen/{instrumen}', [App\Http\Controllers\Pengawas\InstrumenController::class, 'destroy'])->name('instrumen.destroy');
        
        // Instrumen Items Management
        Route::get('/instrumen/{instrumen}/items', [App\Http\Controllers\Pengawas\InstrumenItemController::class, 'index'])->name('items.index');
        Route::get('/instrumen/{instrumen}/items/{item}', [App\Http\Controllers\Pengawas\InstrumenItemController::class, 'show'])->name('items.show');
        Route::get('/instrumen/{instrumen}/items/create', [App\Http\Controllers\Pengawas\InstrumenItemController::class, 'create'])->name('items.create');
        Route::post('/instrumen/{instrumen}/items', [App\Http\Controllers\Pengawas\InstrumenItemController::class, 'store'])->name('items.store');
        Route::get('/instrumen/{instrumen}/items/{item}/edit', [App\Http\Controllers\Pengawas\InstrumenItemController::class, 'edit'])->name('items.edit');
        Route::put('/instrumen/{instrumen}/items/{item}', [App\Http\Controllers\Pengawas\InstrumenItemController::class, 'update'])->name('items.update');
        Route::delete('/instrumen/{instrumen}/items/{item}', [App\Http\Controllers\Pengawas\InstrumenItemController::class, 'destroy'])->name('items.destroy');
        
        // Sub Komponen Management (pengawas hanya bisa lihat)
        Route::get('/instrumen/{instrumen}/items/{item}/sub-komponen', [App\Http\Controllers\Pengawas\SubKomponenController::class, 'index'])->name('items.sub-komponen');
        Route::get('/instrumen/{instrumen}/items/{item}/sub-komponen/{subKomponen}', [App\Http\Controllers\Pengawas\SubKomponenController::class, 'show'])->name('items.sub-komponen.show');
        
        // Distribusi routes
        Route::resource('distribusi', App\Http\Controllers\Pengawas\DistribusiController::class);
        Route::post('/distribusi/{id}/close', [App\Http\Controllers\Pengawas\DistribusiController::class, 'close'])->name('distribusi.close');
        
        // Penilaian Kepala routes
        Route::get('/penilaian', [App\Http\Controllers\Pengawas\PenilaianKepalaController::class, 'index'])->name('penilaian.index');
        Route::get('/penilaian/{id}', [App\Http\Controllers\Pengawas\PenilaianKepalaController::class, 'show'])->name('penilaian.show');
        Route::get('/penilaian/{id}/debug', [App\Http\Controllers\Pengawas\PenilaianKepalaController::class, 'debug'])->name('penilaian.debug');
        Route::put('/penilaian/{id}', [App\Http\Controllers\Pengawas\PenilaianKepalaController::class, 'update'])->name('penilaian.update');
        Route::post('/penilaian/{id}/ubah-status', [App\Http\Controllers\Pengawas\PenilaianKepalaController::class, 'ubahStatus'])->name('penilaian.ubah-status');
        Route::post('/penilaian/{id}/submit', [App\Http\Controllers\Pengawas\PenilaianKepalaController::class, 'submit'])->name('penilaian.submit');

        // Raport generation (pengawas)
        Route::get('/penilaian/{id}/raport', [App\Http\Controllers\Pengawas\RaportController::class, 'generateRaport'])->name('penilaian.raport');
        
        // Profile routes
        Route::get('/profile', [App\Http\Controllers\Pengawas\ProfileController::class, 'index'])->name('profile.index');
        Route::get('/profile/edit', [App\Http\Controllers\Pengawas\ProfileController::class, 'edit'])->name('profile.edit');
        Route::put('/profile', [App\Http\Controllers\Pengawas\ProfileController::class, 'update'])->name('profile.update');
        Route::get('/profile/password', [App\Http\Controllers\Pengawas\ProfileController::class, 'editPassword'])->name('profile.edit-password');
        Route::put('/profile/password', [App\Http\Controllers\Pengawas\ProfileController::class, 'updatePassword'])->name('profile.update-password');
    });
    
    // Kepala routes
    Route::prefix('kepala')->name('kepala.')->middleware(['role:kepala'])->group(function () {
        Route::get('/dashboard', [App\Http\Controllers\Kepala\DashboardController::class, 'index'])->name('dashboard');
        
        // Dokumentasi Kepala
        Route::get('/dokumentasi', function () {
            return view('docs.guide', ['role' => 'kepala']);
        })->name('docs');
        
        // Lembaga Management
        Route::get('/lembaga', [App\Http\Controllers\Kepala\LembagaController::class, 'index'])->name('lembaga.index');
        Route::get('/lembaga/edit', [App\Http\Controllers\Kepala\LembagaController::class, 'edit'])->name('lembaga.edit');
        Route::put('/lembaga/update', [App\Http\Controllers\Kepala\LembagaController::class, 'update'])->name('lembaga.update');
        // Removed: kepala manage guru (tidak diperlukan)
        Route::get('/lembaga/instrumen', [App\Http\Controllers\Kepala\LembagaController::class, 'instrumen'])->name('lembaga.instrumen');
        
        // Instrumen Management
        Route::get('/instrumen', [App\Http\Controllers\Kepala\InstrumenController::class, 'index'])->name('instrumen.index');
        Route::get('/instrumen/{id}', [App\Http\Controllers\Kepala\InstrumenController::class, 'show'])->name('instrumen.show');
        Route::get('/instrumen/{id}/isi', [App\Http\Controllers\Kepala\InstrumenController::class, 'isi'])->name('instrumen.isi');
        Route::post('/instrumen/{id}/isi', [App\Http\Controllers\Kepala\InstrumenController::class, 'simpanIsi'])->name('instrumen.isi.simpan');
        Route::post('/instrumen/{id}/ubah-status', [App\Http\Controllers\Kepala\InstrumenController::class, 'ubahStatus'])->name('instrumen.ubah-status');
        
        // Notification Management
        Route::post('/notifications/{notificationId}/mark-read', [App\Http\Controllers\Kepala\InstrumenController::class, 'markNotificationAsRead'])->name('notifications.mark-read');
        Route::post('/notifications/mark-all-read', [App\Http\Controllers\Kepala\InstrumenController::class, 'markAllNotificationsAsRead'])->name('notifications.mark-all-read');
        Route::get('/notifications', [App\Http\Controllers\Kepala\InstrumenController::class, 'getNotifications'])->name('notifications.get');
        
        // Profile routes
        Route::get('/profile', [App\Http\Controllers\Kepala\ProfileController::class, 'index'])->name('profile.index');
        Route::get('/profile/edit', [App\Http\Controllers\Kepala\ProfileController::class, 'edit'])->name('profile.edit');
        Route::put('/profile', [App\Http\Controllers\Kepala\ProfileController::class, 'update'])->name('profile.update');
        Route::get('/profile/password', [App\Http\Controllers\Kepala\ProfileController::class, 'editPassword'])->name('profile.edit-password');
        Route::put('/profile/password', [App\Http\Controllers\Kepala\ProfileController::class, 'updatePassword'])->name('profile.update-password');
        
        // Removed: kepala melakukan review penilaian guru (alur baru pengawas menilai kepala)
        
        // Kepala view-only hasil penilaian kepala
        Route::get('/penilaian', [App\Http\Controllers\Kepala\PenilaianKepalaController::class, 'index'])->name('penilaian.index');
        Route::get('/penilaian/{id}', [App\Http\Controllers\Kepala\PenilaianKepalaController::class, 'show'])->name('penilaian.show');
    });
    
    // Removed: Guru routes group (role dan fitur guru ditiadakan)
});
