@extends('layouts.app')

@section('title', 'Edit Data Lembaga')

@section('breadcrumb')
<li class="breadcrumb-item"><a href="{{ route('kepala.lembaga.index') }}">Lembaga Saya</a></li>
<li class="breadcrumb-item active">Edit</li>
@endsection

@section('content')
<div class="card">
    <div class="card-header">
        <h3 class="card-title">
            <i class="fas fa-edit"></i> Edit Data Lembaga: {{ $lembaga->nama_lembaga }}
            <small class="text-muted">(ID: {{ $lembaga->id }})</small>
        </h3>
        <div class="card-tools">
            <a href="{{ route('kepala.lembaga.index') }}" class="btn btn-secondary btn-sm">
                <i class="fas fa-arrow-left"></i> Kembali
            </a>
        </div>
    </div>
    <div class="card-body">
        @if (session('success'))
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                {{ session('success') }}
                <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
        @endif

        @if ($errors->any())
            <div class="alert alert-danger">
                <ul class="mb-0">
                    @foreach ($errors->all() as $error)
                        <li>{{ $error }}</li>
                    @endforeach
                </ul>
            </div>
        @endif

        <form action="{{ route('kepala.lembaga.update') }}" method="POST" enctype="multipart/form-data">
            @csrf
            @method('PUT')

            <!-- Basic Information -->
            <div class="card card-primary card-outline">
                <div class="card-header">
                    <h3 class="card-title">Informasi Dasar</h3>
                    <div class="card-tools">
                        <span class="badge badge-info">ID Lembaga: {{ $lembaga->id }}</span>
                    </div>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="npsn_nsm">NSM <span class="text-danger">*</span></label>
                                <input type="text" class="form-control @error('npsn_nsm') is-invalid @enderror" 
                                       id="npsn_nsm" name="npsn_nsm" value="{{ old('npsn_nsm', $lembaga->npsn_nsm) }}" 
                                       required readonly>
                                <small class="form-text text-muted">Nomor Statistik Madrasah (tidak dapat diubah)</small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="nama_lembaga">Nama Lembaga <span class="text-danger">*</span></label>
                                <input type="text" class="form-control @error('nama_lembaga') is-invalid @enderror" 
                                       id="nama_lembaga" name="nama_lembaga" value="{{ old('nama_lembaga', $lembaga->nama_lembaga) }}" 
                                       required>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="jenis">Jenis Lembaga <span class="text-danger">*</span></label>
                                <select class="form-control @error('jenis') is-invalid @enderror" id="jenis" name="jenis" required readonly>
                                    <option value="RA" {{ old('jenis', $lembaga->jenis) == 'RA' ? 'selected' : '' }}>RA (Raudhatul Athfal)</option>
                                    <option value="MI" {{ old('jenis', $lembaga->jenis) == 'MI' ? 'selected' : '' }}>MI (Madrasah Ibtidaiyah)</option>
                                    <option value="MTs" {{ old('jenis', $lembaga->jenis) == 'MTs' ? 'selected' : '' }}>MTs (Madrasah Tsanawiyah)</option>
                                    <option value="MA" {{ old('jenis', $lembaga->jenis) == 'MA' ? 'selected' : '' }}>MA (Madrasah Aliyah)</option>
                                </select>
                                <small class="form-text text-muted">Jenis lembaga (tidak dapat diubah)</small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="tahun_berdiri">Tahun Berdiri</label>
                                <input type="number" class="form-control @error('tahun_berdiri') is-invalid @enderror" 
                                       id="tahun_berdiri" name="tahun_berdiri" value="{{ old('tahun_berdiri', $lembaga->tahun_berdiri) }}" 
                                       min="1900" max="{{ date('Y') }}">
                            </div>
                        </div>
                    </div>

                    <!-- Logo Upload Section -->
                    <div class="row">
                        <div class="col-md-12">
                            <div class="form-group">
                                <label for="logo_lembaga">Logo Lembaga</label>
                                <input type="file" class="form-control @error('logo_lembaga') is-invalid @enderror" 
                                       id="logo_lembaga" name="logo_lembaga" accept="image/*">
                                @error('logo_lembaga')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="form-text text-muted">Format: JPEG, PNG, JPG, GIF, WEBP. Maksimal 5MB.</small>
                                @if($lembaga->logo_lembaga)
                                    <div class="mt-2">
                                        <strong>Logo Saat Ini:</strong><br>
                                        <img src="{{ Storage::url($lembaga->logo_lembaga) }}" 
                                             alt="Logo {{ $lembaga->nama_lembaga }}" 
                                             class="img-thumbnail" 
                                             style="max-height: 100px; max-width: 150px;">
                                        <br><small class="text-muted">Pilih file baru untuk mengganti logo</small>
                                    </div>
                                @endif
                            </div>
                        </div>
                    </div>

                    <!-- Information about pengawas -->
                    @if($lembaga->pengawas)
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i>
                        <strong>Pengawas:</strong> {{ $lembaga->pengawas->name }}
                        <br>
                        <small class="text-muted">Data yang Anda edit akan terlihat oleh pengawas dan admin.</small>
                    </div>
                    @else
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle"></i>
                        <strong>Belum ada pengawas yang diassign.</strong>
                        <br>
                        <small class="text-muted">Hubungi admin untuk mengassign pengawas ke lembaga ini.</small>
                    </div>
                    @endif
                    </div>
                </div>
            </div>

            <!-- Address Information -->
            <div class="card card-success card-outline">
                <div class="card-header">
                    <h3 class="card-title">Informasi Alamat</h3>
                </div>
                <div class="card-body">
                    <div class="form-group">
                        <label for="alamat_lengkap">Alamat Lengkap <span class="text-danger">*</span></label>
                        <textarea class="form-control @error('alamat_lengkap') is-invalid @enderror" 
                                  id="alamat_lengkap" name="alamat_lengkap" rows="3" required>{{ old('alamat_lengkap', $lembaga->alamat_lengkap) }}</textarea>
                    </div>

                    <div class="row">
                        <div class="col-md-3">
                            <div class="form-group">
                                <label for="provinsi">Provinsi <span class="text-danger">*</span></label>
                                <select class="form-control @error('provinsi') is-invalid @enderror" 
                                        id="provinsi" name="provinsi" required>
                                    <option value="">Pilih Provinsi...</option>
                                    @if($lembaga->provinsi)
                                        <option value="{{ $lembaga->provinsi }}" selected>{{ $lembaga->provinsi_name ?? $lembaga->provinsi }}</option>
                                    @endif
                                </select>
                                @error('provinsi')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="form-group">
                                <label for="kabupaten">Kabupaten <span class="text-danger">*</span></label>
                                <select class="form-control @error('kabupaten') is-invalid @enderror" 
                                        id="kabupaten" name="kabupaten" required>
                                    <option value="">Pilih Kabupaten...</option>
                                    @if($lembaga->kabupaten)
                                        <option value="{{ $lembaga->kabupaten }}" selected>{{ $lembaga->kabupaten_name ?? $lembaga->kabupaten }}</option>
                                    @endif
                                </select>
                                @error('kabupaten')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="form-group">
                                <label for="kecamatan">Kecamatan <span class="text-danger">*</span></label>
                                <select class="form-control @error('kecamatan') is-invalid @enderror" 
                                        id="kecamatan" name="kecamatan" required>
                                    <option value="">Pilih Kecamatan...</option>
                                    @if($lembaga->kecamatan)
                                        <option value="{{ $lembaga->kecamatan }}" selected>{{ $lembaga->kecamatan_name ?? $lembaga->kecamatan }}</option>
                                    @endif
                                </select>
                                @error('kecamatan')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="form-group">
                                <label for="kelurahan">Kelurahan <span class="text-danger">*</span></label>
                                <select class="form-control @error('kelurahan') is-invalid @enderror" 
                                        id="kelurahan" name="kelurahan" required>
                                    <option value="">Pilih Kelurahan...</option>
                                    @if($lembaga->kelurahan)
                                        <option value="{{ $lembaga->kelurahan }}" selected>{{ $lembaga->kelurahan_name ?? $lembaga->kelurahan }}</option>
                                    @endif
                                </select>
                                @error('kelurahan')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="kode_pos">Kode Pos</label>
                                <input type="text" class="form-control @error('kode_pos') is-invalid @enderror" 
                                       id="kode_pos" name="kode_pos" value="{{ old('kode_pos', $lembaga->kode_pos) }}" 
                                       maxlength="10" placeholder="Opsional">
                                <small class="form-text text-muted">
                                    <i class="fas fa-info-circle"></i> Jika tersedia, kode pos akan terisi otomatis berdasarkan kelurahan. Anda juga dapat mengisi/menyesuaikan secara manual (opsional).
                                </small>
                                @error('kode_pos')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="telepon">Telepon</label>
                                <input type="text" class="form-control @error('telepon') is-invalid @enderror" 
                                       id="telepon" name="telepon" value="{{ old('telepon', $lembaga->telepon) }}">
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Contact Information -->
            <div class="card card-info card-outline">
                <div class="card-header">
                    <h3 class="card-title">Informasi Kontak</h3>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="email">Email Lembaga</label>
                                <input type="email" class="form-control @error('email') is-invalid @enderror" 
                                       id="email" name="email" value="{{ old('email', $lembaga->email) }}">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="website">Website</label>
                                <input type="url" class="form-control @error('website') is-invalid @enderror" 
                                       id="website" name="website" value="{{ old('website', $lembaga->website) }}" 
                                       placeholder="https://example.com">
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Head Information -->
            <div class="card card-warning card-outline">
                <div class="card-header">
                    <h3 class="card-title">Informasi Kepala Lembaga</h3>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="nama_kepala">Nama Kepala</label>
                                <input type="text" class="form-control" 
                                       id="nama_kepala" value="{{ $lembaga->kepala->name ?? $lembaga->nama_kepala ?? 'Belum ada kepala' }}" readonly>
                                <small class="form-text text-muted">
                                    <i class="fas fa-info-circle"></i> Nama kepala akan otomatis terupdate dari akun yang diassign admin.
                                </small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="nip_kepala">NIP Kepala <span class="text-danger">*</span></label>
                                <input type="text" class="form-control @error('nip_kepala') is-invalid @enderror" 
                                       id="nip_kepala" name="nip_kepala" value="{{ old('nip_kepala', $lembaga->nip_kepala) }}" required>
                            </div>
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="akreditasi">Akreditasi</label>
                        <select class="form-control @error('akreditasi') is-invalid @enderror" id="akreditasi" name="akreditasi">
                            <option value="">Pilih Akreditasi</option>
                            <option value="A" {{ old('akreditasi', $lembaga->akreditasi) == 'A' ? 'selected' : '' }}>A (Sangat Baik)</option>
                            <option value="B" {{ old('akreditasi', $lembaga->akreditasi) == 'B' ? 'selected' : '' }}>B (Baik)</option>
                            <option value="C" {{ old('akreditasi', $lembaga->akreditasi) == 'C' ? 'selected' : '' }}>C (Cukup)</option>
                            <option value="TT" {{ old('akreditasi', $lembaga->akreditasi) == 'TT' ? 'selected' : '' }}>TT (Terakreditasi)</option>
                        </select>
                    </div>
                </div>
            </div>

            <div class="form-group">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i> Simpan Perubahan
                </button>
                <a href="{{ route('kepala.lembaga.index') }}" class="btn btn-secondary">
                    <i class="fas fa-times"></i> Batal
                </a>
            </div>
        </form>
    </div>
</div>

<!-- Hidden inputs to store current values -->
<input type="hidden" id="current_provinsi" value="{{ old('provinsi', $lembaga->provinsi) }}">
<input type="hidden" id="current_kabupaten" value="{{ old('kabupaten', $lembaga->kabupaten) }}">
<input type="hidden" id="current_kecamatan" value="{{ old('kecamatan', $lembaga->kecamatan) }}">
<input type="hidden" id="current_kelurahan" value="{{ old('kelurahan', $lembaga->kelurahan) }}">
@endsection

@push('scripts')
<script>
$(document).ready(function() {
    // Logo preview functionality
    $('#logo_lembaga').on('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
                // Create or update preview
                let preview = $('#logo-preview');
                if (preview.length === 0) {
                    preview = $('<div id="logo-preview" class="mt-2"><strong>Preview Logo Baru:</strong><br><img class="img-thumbnail" style="max-height: 100px; max-width: 150px;"></div>');
                    $('#logo_lembaga').after(preview);
                }
                preview.find('img').attr('src', e.target.result);
            };
            reader.readAsDataURL(file);
        }
    });
    const currentProvinsi = $('#current_provinsi').val();
    const currentKabupaten = $('#current_kabupaten').val();
    const currentKecamatan = $('#current_kecamatan').val();
    const currentKelurahan = $('#current_kelurahan').val();


    // Load provinces on page load
    loadProvinces();

    // Load provinces function
    function loadProvinces() {
        $.get('/api/wilayah/provinsi', function(data) {
            const provinsiSelect = $('#provinsi');
            provinsiSelect.empty().append('<option value="">Pilih Provinsi</option>');
            
            // Add all provinces to dropdown
            $.each(data, function(index, provinsi) {
                const selected = (provinsi.id == currentProvinsi) ? 'selected' : '';
                provinsiSelect.append(`<option value="${provinsi.id}" ${selected}>${provinsi.name}</option>`);
            });

            // Auto-load kabupaten if current value exists
            if (currentProvinsi) {
                setTimeout(function() {
                    loadKabupaten(currentProvinsi);
                }, 100);
            }
        }).fail(function(xhr, status, error) {
            // Silent fail - no console logging
        });
    }

    // Load kabupaten function
    function loadKabupaten(provinsiId) {
        if (!provinsiId) {
            $('#kabupaten').empty().append('<option value="">Pilih Kabupaten</option>');
            $('#kecamatan').empty().append('<option value="">Pilih Kecamatan</option>');
            $('#kelurahan').empty().append('<option value="">Pilih Kelurahan</option>');
            return;
        }

        $.get('/api/wilayah/kabupaten', {province_id: provinsiId}, function(data) {
            const kabupatenSelect = $('#kabupaten');
            kabupatenSelect.empty().append('<option value="">Pilih Kabupaten</option>');
            
            // Add all kabupaten to dropdown
            $.each(data, function(index, kabupaten) {
                const selected = (kabupaten.id == currentKabupaten) ? 'selected' : '';
                kabupatenSelect.append(`<option value="${kabupaten.id}" ${selected}>${kabupaten.name}</option>`);
            });

            // Auto-load kecamatan if current value exists
            if (currentKabupaten) {
                setTimeout(function() {
                    loadKecamatan(currentKabupaten);
                }, 100);
            }
        }).fail(function(xhr, status, error) {
            // Silent fail - no console logging
        });
    }

    // Load kecamatan function
    function loadKecamatan(kabupatenId) {
        if (!kabupatenId) {
            $('#kecamatan').empty().append('<option value="">Pilih Kecamatan</option>');
            $('#kelurahan').empty().append('<option value="">Pilih Kelurahan</option>');
            return;
        }

        $.get('/api/wilayah/kecamatan', {regency_id: kabupatenId}, function(data) {
            const kecamatanSelect = $('#kecamatan');
            kecamatanSelect.empty().append('<option value="">Pilih Kecamatan</option>');
            
            // Add all kecamatan to dropdown
            $.each(data, function(index, kecamatan) {
                const selected = (kecamatan.id == currentKecamatan) ? 'selected' : '';
                kecamatanSelect.append(`<option value="${kecamatan.id}" ${selected}>${kecamatan.name}</option>`);
            });

            // Auto-load kelurahan if current value exists
            if (currentKecamatan) {
                setTimeout(function() {
                    loadKelurahan(currentKecamatan);
                }, 100);
            }
        }).fail(function(xhr, status, error) {
            // Silent fail - no console logging
        });
    }

    // Load kelurahan function
    function loadKelurahan(kecamatanId) {
        if (!kecamatanId) {
            $('#kelurahan').empty().append('<option value="">Pilih Kelurahan</option>');
            return;
        }

        $.get('/api/wilayah/kelurahan', {district_id: kecamatanId}, function(data) {
            const kelurahanSelect = $('#kelurahan');
            kelurahanSelect.empty().append('<option value="">Pilih Kelurahan</option>');
            
            // Add all kelurahan to dropdown
            $.each(data, function(index, kelurahan) {
                const selected = (kelurahan.id == currentKelurahan) ? 'selected' : '';
                kelurahanSelect.append(`<option value="${kelurahan.id}" ${selected}>${kelurahan.name}</option>`);
            });

            // Auto-select current kelurahan if exists
            if (currentKelurahan) {
                setTimeout(function() {
                    $('#kelurahan').val(currentKelurahan);
                }, 100);
            }
        }).fail(function(xhr, status, error) {
            // Silent fail - no console logging
        });
    }

    // Event handlers for cascading dropdowns
    $('#provinsi').on('change', function() {
        const provinsiId = $(this).val();
        loadKabupaten(provinsiId);
        $('#kecamatan').empty().append('<option value="">Pilih Kecamatan</option>');
        $('#kelurahan').empty().append('<option value="">Pilih Kelurahan</option>');
    });

    $('#kabupaten').on('change', function() {
        const kabupatenId = $(this).val();
        loadKecamatan(kabupatenId);
        $('#kelurahan').empty().append('<option value="">Pilih Kelurahan</option>');
    });

    $('#kecamatan').on('change', function() {
        const kecamatanId = $(this).val();
        loadKelurahan(kecamatanId);
    });
});
</script>
@endpush
