<?php

namespace App\Services;

use App\Models\PenilaianGuru;
use App\Models\JawabanGuru;
use App\Models\AppSettings;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class AIAnalysisService
{
    public $settings;
    private $geminiApiKey;
    public $geminiApiUrl;

    public function __construct()
    {
        $this->settings = AppSettings::getSettings();
        $this->geminiApiKey = $this->settings->gemini_api_key;
        // Use different API versions based on model
        $model = $this->settings->gemini_model;
        if (strpos($model, '2.5') !== false) {
            // For Gemini 2.5 models, use v1 API
            $this->geminiApiUrl = 'https://generativelanguage.googleapis.com/v1/models/' . $model . ':generateContent';
        } elseif (strpos($model, '1.5') !== false) {
            // For Gemini 1.5 models, use v1 API
            $this->geminiApiUrl = 'https://generativelanguage.googleapis.com/v1/models/' . $model . ':generateContent';
        } else {
            // For legacy models, use v1beta API
            $this->geminiApiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . $model . ':generateContent';
        }
    }

    /**
     * Check if Gemini AI is enabled and configured
     */
    public function isEnabled(): bool
    {
        return $this->settings->gemini_enabled && !empty($this->geminiApiKey);
    }

    /**
     * Generate AI analysis for kepala madrasah
     */
    public function generateKepalaAnalysis(PenilaianGuru $penilaianGuru): string
    {
        if (!$this->isEnabled()) {
            return 'Analisis AI tidak tersedia. Silakan hubungi administrator untuk mengaktifkan fitur ini.';
        }

        $data = $this->prepareAnalysisData($penilaianGuru);
        
        $prompt = $this->buildKepalaPrompt($data);
        
        $aiResult = $this->callGeminiAPI($prompt);
        
        // If AI fails, return manual analysis
        if (strpos($aiResult, 'Terjadi kesalahan') !== false) {
            return $this->generateManualAnalysis($data);
        }
        
        return $aiResult;
    }

    /**
     * Generate AI analysis for pengawas
     */
    public function generatePengawasAnalysis(PenilaianGuru $penilaianGuru): string
    {
        if (!$this->isEnabled()) {
            return 'Analisis AI tidak tersedia. Silakan hubungi administrator untuk mengaktifkan fitur ini.';
        }

        $data = $this->prepareAnalysisData($penilaianGuru);
        
        $prompt = $this->buildPengawasPrompt($data);
        
        return $this->callGeminiAPI($prompt);
    }

    /**
     * Prepare data for AI analysis
     */
    public function prepareAnalysisData(PenilaianGuru $penilaianGuru): array
    {
        $penilaianGuru->load([
            'guru',
            'kepala',
            'distribusiInstrumen.instrumen',
            'distribusiInstrumen.komponen',
            'jawabanGuru.instrumenItem.subKomponen.komponenPenilaian'
        ]);

        $data = [
            'guru' => $penilaianGuru->guru,
            'kepala' => $penilaianGuru->kepala,
            'instrumen' => $penilaianGuru->distribusiInstrumen->instrumen,
            'komponen' => $penilaianGuru->distribusiInstrumen->komponen,
            'total_nilai' => $penilaianGuru->total_nilai,
            'rata_rata' => $penilaianGuru->rata_rata,
            'sub_komponen_analysis' => []
        ];

        // Group jawaban by sub komponen
        $jawabanBySubKomponen = $penilaianGuru->jawabanGuru->groupBy('instrumenItem.sub_komponen_id');

        foreach ($jawabanBySubKomponen as $subKomponenId => $jawabanGroup) {
            $subKomponen = $jawabanGroup->first()->instrumenItem->subKomponen;
            $nilaiKepala = $jawabanGroup->pluck('nilai_kepala')->filter()->values();
            
            if ($nilaiKepala->isNotEmpty()) {
                $rataRata = $nilaiKepala->avg();
                $kategori = $this->determineCategory($rataRata);
                
                $data['sub_komponen_analysis'][] = [
                    'nama_sub_komponen' => $subKomponen->nama_sub_komponen,
                    'deskripsi' => $subKomponen->deskripsi,
                    'rata_rata' => round($rataRata, 2),
                    'kategori' => $kategori,
                    'nilai_detail' => $nilaiKepala->toArray(),
                    'indikator_kinerja' => $jawabanGroup->map(function($jawaban) {
                        return [
                            'indikator' => $jawaban->instrumenItem->indikator_kinerja,
                            'nilai' => $jawaban->nilai_kepala,
                            'bukti' => $jawaban->keterangan_bukti
                        ];
                    })->toArray()
                ];
            }
        }

        return $data;
    }

    /**
     * Build prompt for kepala Madrasah analysis
     */
    private function buildKepalaPrompt(array $data): string
    {
        $subKomponenText = '';
        foreach ($data['sub_komponen_analysis'] as $analysis) {
            $subKomponenText .= "\nSub Komponen: {$analysis['nama_sub_komponen']}\n";
            $subKomponenText .= "- Nilai Rata-rata: {$analysis['rata_rata']}/5.00\n";
            $subKomponenText .= "- Kategori: {$analysis['kategori']}\n";
            $subKomponenText .= "- Deskripsi: {$analysis['deskripsi']}\n";
        }

        return "Anda adalah seorang kepala madrasah yang berpengalaman dengan latar belakang pendidikan yang kuat. Analisis hasil penilaian kinerja guru berikut:

DATA PENILAIAN:
Nama Guru: {$data['guru']->nama_lengkap}
Nama Kepala: {$data['kepala']->name}
Instrumen: " . ($data['instrumen']->nama_instrumen ?: 'N/A') . "
Komponen: {$data['komponen']->nama_komponen}
Total Nilai: {$data['total_nilai']}
Rata-rata Keseluruhan: {$data['rata_rata']}/5.00

ANALISIS PER SUB KOMPONEN:
{$subKomponenText}

TUGAS ANDA:
1. Berikan analisis mendalam per sub komponen dengan format yang jelas
2. Identifikasi kekuatan dan area perbaikan berdasarkan data
3. Buat rencana tindak lanjut yang konkret dan actionable
4. Berikan rekomendasi strategis dari perspektif kepala madrasah

FORMAT OUTPUT:
RANGKUMAN HASIL PENILAIAN KINERJA GURU

ANALISIS PER SUB KOMPONEN:
[Analisis detail per sub komponen]

RINGKASAN KESELURUHAN:
- Total Nilai: [nilai]/[maksimal]
- Rata-rata Keseluruhan: [rata-rata]/5.00
- Kategori Penilaian: [Sangat Baik/Baik/Cukup/Kurang]

KEKUATAN YANG DITEMUKAN:
[List kekuatan]

AREA YANG PERLU DITINGKATKAN:
[List area perbaikan]

RENCANA TINDAK LANJUT:
1. [Rekomendasi spesifik]
2. [Program pelatihan]
3. [Timeline implementasi]
4. [Monitoring dan evaluasi]

REKOMENDASI KEPALA MADRASAH:
[Kesimpulan dan arahan strategis]

TONALITAS: Profesional, supportive, konstruktif, berbasis data, memberikan arahan yang jelas.

PENTING: Jangan gunakan markdown formatting seperti ** atau * dalam output. Gunakan format text biasa saja.";
    }

    /**
     * Build prompt for pengawas analysis
     */
    private function buildPengawasPrompt(array $data): string
    {
        $subKomponenText = '';
        foreach ($data['sub_komponen_analysis'] as $analysis) {
            $subKomponenText .= "\nSub Komponen: {$analysis['nama_sub_komponen']}\n";
            $subKomponenText .= "- Nilai Rata-rata: {$analysis['rata_rata']}/5.00\n";
            $subKomponenText .= "- Kategori: {$analysis['kategori']}\n";
            $subKomponenText .= "- Deskripsi: {$analysis['deskripsi']}\n";
        }

        return "Anda adalah seorang pengawas madrasah yang berpengalaman, kompeten, dan menguasai analisis instrumen penilaian. Analisis hasil penilaian kinerja guru berikut:

DATA PENILAIAN:
Nama Guru: {$data['guru']->nama_lengkap}
Nama Kepala: {$data['kepala']->name}
Instrumen: " . ($data['instrumen']->nama_instrumen ?: 'N/A') . "
Komponen: {$data['komponen']->nama_komponen}
Total Nilai: {$data['total_nilai']}
Rata-rata Keseluruhan: {$data['rata_rata']}/5.00

ANALISIS PER SUB KOMPONEN:
{$subKomponenText}

TUGAS ANDA:
1. Berikan analisis objektif per sub komponen dari perspektif pengawas
2. Evaluasi kualitas penilaian yang dilakukan kepala madrasah
3. Identifikasi area yang memerlukan perhatian khusus
4. Berikan rekomendasi pengawasan dan pembinaan

FORMAT OUTPUT:
EVALUASI PENGAWAS TERHADAP HASIL PENILAIAN

ANALISIS OBJEKTIF PER SUB KOMPONEN:
[Analisis detail per sub komponen dari perspektif pengawas]

EVALUASI KESELURUHAN:
- Validitas Penilaian: [Evaluasi kualitas penilaian]
- Kategori Penilaian: [Sangat Baik/Baik/Cukup/Kurang]
- Tingkat Pencapaian: [Analisis pencapaian]

ASPEK YANG MENONJOL:
[List aspek positif]

AREA YANG MEMERLUKAN PERHATIAN:
[List area yang perlu perhatian khusus]

REKOMENDASI PENGAWASAN:
1. [Rekomendasi pembinaan]
2. [Program pengembangan]
3. [Monitoring berkelanjutan]
4. [Evaluasi tindak lanjut]

KESIMPULAN PENGAWAS:
[Kesimpulan dan arahan dari perspektif pengawas]

TONALITAS: Objektif, analitis, professional, memberikan evaluasi yang konstruktif.

PENTING: Jangan gunakan markdown formatting seperti ** atau * dalam output. Gunakan format text biasa saja.";
    }

    /**
     * Call Gemini API
     */
    private function callGeminiAPI(string $prompt): string
    {
        try {
            Log::info('Gemini API Request:', [
                'url' => $this->geminiApiUrl,
                'api_key_length' => strlen($this->geminiApiKey),
                'prompt_length' => strlen($prompt),
                'model' => $this->settings->gemini_model
            ]);

            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
            ])->post($this->geminiApiUrl . '?key=' . $this->geminiApiKey, [
                'contents' => [
                    [
                        'parts' => [
                            ['text' => $prompt]
                        ]
                    ]
                ],
                'generationConfig' => [
                    'temperature' => $this->settings->gemini_temperature,
                    'topK' => 40,
                    'topP' => 0.95,
                    'maxOutputTokens' => $this->settings->gemini_max_tokens,
                ]
            ]);

            Log::info('Gemini API Response:', [
                'status' => $response->status(),
                'successful' => $response->successful(),
                'body' => $response->body()
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $result = $data['candidates'][0]['content']['parts'][0]['text'] ?? 'Analisis tidak dapat dihasilkan.';
                Log::info('Gemini API Success:', ['result_length' => strlen($result)]);
                return $result;
            }

            Log::error('Gemini API Error:', [
                'status' => $response->status(),
                'body' => $response->body(),
                'headers' => $response->headers()
            ]);
            return 'Terjadi kesalahan dalam menghasilkan analisis. Silakan coba lagi.';

        } catch (\Exception $e) {
            Log::error('Gemini API Exception:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return 'Terjadi kesalahan dalam menghasilkan analisis. Silakan coba lagi.';
        }
    }

    /**
     * Generate manual analysis when AI fails
     */
    public function generateManualAnalysis(array $data): string
    {
        $guru = $data['guru'];
        $totalNilai = $data['total_nilai'];
        $rataRata = $data['rata_rata'];
        $subKomponenAnalysis = $data['sub_komponen_analysis'];
        
        $kategori = $this->determineCategory($rataRata ?? 0);
        
        $analysis = "ANALISIS KINERJA GURU\n\n";
        $analysis .= "Berdasarkan hasil penilaian kinerja guru " . $guru->nama_lengkap . " dengan total nilai " . $totalNilai . " dan rata-rata " . $rataRata . "/5.00, dapat disimpulkan bahwa kinerja guru berada dalam kategori " . $kategori . ".\n\n";
        
        $analysis .= "RINCIAN PENILAIAN PER SUB KOMPONEN:\n";
        foreach ($subKomponenAnalysis as $index => $subKomponen) {
            $analysis .= ($index + 1) . ". " . $subKomponen['nama_sub_komponen'] . ": " . $subKomponen['rata_rata'] . "/5.00 (" . $subKomponen['kategori'] . ")\n";
            
            // Add detailed analysis for each sub komponen
            $analysis .= "   Analisis: ";
            if ($subKomponen['rata_rata'] >= 4.0) {
                $analysis .= "Guru menunjukkan kinerja yang sangat baik dalam " . $subKomponen['nama_sub_komponen'] . ". ";
            } elseif ($subKomponen['rata_rata'] >= 3.0) {
                $analysis .= "Guru menunjukkan kinerja yang baik dalam " . $subKomponen['nama_sub_komponen'] . " dengan beberapa area yang bisa ditingkatkan. ";
            } else {
                $analysis .= "Guru memerlukan perhatian khusus dalam " . $subKomponen['nama_sub_komponen'] . ". ";
            }
            $analysis .= "Rekomendasi: Fokus pada penguatan aspek yang masih lemah dan pengembangan strategi yang lebih efektif.\n\n";
        }
        
        $analysis .= "\nKEKUATAN:\n";
        if (($rataRata ?? 0) >= 4.0) {
            $analysis .= "• Guru menunjukkan kinerja yang sangat baik dalam semua aspek penilaian\n";
            $analysis .= "• Komitmen dan dedikasi yang tinggi dalam melaksanakan tugas\n";
        } elseif (($rataRata ?? 0) >= 3.0) {
            $analysis .= "• Guru menunjukkan kinerja yang baik dalam sebagian besar aspek\n";
            $analysis .= "• Ada potensi untuk pengembangan lebih lanjut\n";
        } else {
            $analysis .= "• Guru menunjukkan upaya dalam melaksanakan tugas\n";
            $analysis .= "• Memerlukan bimbingan dan pendampingan intensif\n";
        }
        
        $analysis .= "\nAREA PENGEMBANGAN:\n";
        if (($rataRata ?? 0) < 3.0) {
            $analysis .= "• Perlu peningkatan dalam perencanaan pembelajaran\n";
            $analysis .= "• Perlu penguatan dalam pelaksanaan pembelajaran\n";
            $analysis .= "• Perlu peningkatan dalam evaluasi pembelajaran\n";
        } else {
            $analysis .= "• Terus mempertahankan kinerja yang sudah baik\n";
            $analysis .= "• Mengembangkan inovasi dalam pembelajaran\n";
        }
        
        $analysis .= "\nRENCANA TINDAK LANJUT:\n";
        if (($rataRata ?? 0) < 3.0) {
            $analysis .= "• Memberikan pendampingan intensif oleh kepala madrasah\n";
            $analysis .= "• Mengikuti pelatihan dan workshop pengembangan kompetensi\n";
            $analysis .= "• Melakukan evaluasi berkala setiap bulan\n";
        } else {
            $analysis .= "• Memberikan apresiasi atas kinerja yang baik\n";
            $analysis .= "• Memberikan kesempatan untuk berbagi pengalaman dengan guru lain\n";
            $analysis .= "• Melanjutkan pengembangan profesional berkelanjutan\n";
        }
        
        return $analysis;
    }

    /**
     * Determine category based on score
     */
    private function determineCategory(float $score): string
    {
        if ($score >= 4.5) return 'Sangat Baik';
        if ($score >= 3.5) return 'Baik';
        if ($score >= 2.5) return 'Cukup';
        return 'Kurang';
    }

    /**
     * Generate AI analysis for pengawas evaluating kepala (comprehensive)
     */
    public function generatePengawasAnalysisKepala(\App\Models\PenilaianKepala $penilaian): string
    {
        if (!$this->isEnabled()) {
            return $this->generateManualAnalysisKepala($penilaian);
        }

        $data = $this->prepareAnalysisDataKepala($penilaian);
        $prompt = $this->buildPengawasAnalysisKepalaPrompt($data);
        
        $aiResult = $this->callGeminiAPI($prompt);
        
        // If AI fails, return manual analysis
        if (strpos($aiResult, 'Terjadi kesalahan') !== false) {
            return $this->generateManualAnalysisKepala($penilaian);
        }
        
        return $aiResult;
    }

    /**
     * Prepare data for AI analysis of kepala by pengawas
     */
    private function prepareAnalysisDataKepala(\App\Models\PenilaianKepala $penilaian): array
    {
        $penilaian->load([
            'kepala',
            'pengawas',
            'distribusiInstrumen.instrumen',
            'distribusiInstrumen.komponen',
            'jawabanKepala.instrumenItem.subKomponen.komponenPenilaian'
        ]);

        // Calculate totals
        $jawaban = $penilaian->jawabanKepala;
        $totalNilaiPengawas = $jawaban->sum('nilai_pengawas');
        $totalNilaiKepala = $jawaban->sum('nilai_guru_self');
        $jumlahItemTerisiPengawas = $jawaban->whereNotNull('nilai_pengawas')->where('nilai_pengawas', '>', 0)->count();
        $jumlahItemTerisiKepala = $jawaban->whereNotNull('nilai_guru_self')->where('nilai_guru_self', '>', 0)->count();
        $rataRataPengawas = $jumlahItemTerisiPengawas > 0 ? round($totalNilaiPengawas / $jumlahItemTerisiPengawas, 2) : 0;
        $rataRataKepala = $jumlahItemTerisiKepala > 0 ? round($totalNilaiKepala / $jumlahItemTerisiKepala, 2) : 0;

        $data = [
            'kepala' => $penilaian->kepala,
            'pengawas' => $penilaian->pengawas,
            'instrumen' => $penilaian->distribusiInstrumen?->instrumen,
            'komponen' => $penilaian->distribusiInstrumen?->komponen,
            'total_nilai_pengawas' => $totalNilaiPengawas,
            'total_nilai_kepala' => $totalNilaiKepala,
            'rata_rata_pengawas' => $rataRataPengawas,
            'rata_rata_kepala' => $rataRataKepala,
            'jumlah_item' => $jawaban->count(),
            'jumlah_terisi_pengawas' => $jumlahItemTerisiPengawas,
            'jumlah_terisi_kepala' => $jumlahItemTerisiKepala,
            'sub_komponen_analysis' => []
        ];

        // Group jawaban by sub komponen
        $jawabanBySubKomponen = $jawaban->groupBy(function($j) {
            return $j->instrumenItem?->sub_komponen_id ?? 'unknown';
        });

        foreach ($jawabanBySubKomponen as $subKomponenId => $jawabanGroup) {
            if ($subKomponenId === 'unknown' || !$jawabanGroup->first()->instrumenItem) {
                continue;
            }

            $subKomponen = $jawabanGroup->first()->instrumenItem->subKomponen;
            $nilaiPengawas = $jawabanGroup->pluck('nilai_pengawas')->filter(function($v) {
                return $v !== null && $v !== '' && $v > 0;
            })->values();
            $nilaiKepala = $jawabanGroup->pluck('nilai_guru_self')->filter(function($v) {
                return $v !== null && $v !== '' && $v > 0;
            })->values();
            
            if ($nilaiPengawas->isNotEmpty()) {
                $rataRataPengawasSub = $nilaiPengawas->avg();
                $rataRataKepalaSub = $nilaiKepala->isNotEmpty() ? $nilaiKepala->avg() : null;
                $kategori = $this->determineCategory($rataRataPengawasSub);
                
                $data['sub_komponen_analysis'][] = [
                    'nama_sub_komponen' => $subKomponen->nama_sub_komponen,
                    'deskripsi' => $subKomponen->deskripsi,
                    'rata_rata_pengawas' => round($rataRataPengawasSub, 2),
                    'rata_rata_kepala' => $rataRataKepalaSub ? round($rataRataKepalaSub, 2) : null,
                    'kategori' => $kategori,
                    'jumlah_item' => $jawabanGroup->count(),
                    'indikator_kinerja' => $jawabanGroup->map(function($jawaban) {
                        return [
                            'indikator' => $jawaban->instrumenItem->indikator_kinerja,
                            'evidensi' => $jawaban->instrumenItem->evidensi_kinerja,
                            'nilai_kepala' => $jawaban->nilai_guru_self,
                            'nilai_pengawas' => $jawaban->nilai_pengawas,
                            'catatan_pengawas' => $jawaban->catatan_pengawas,
                            'bukti' => $jawaban->keterangan_bukti,
                            'url_bukti' => $jawaban->url_bukti,
                        ];
                    })->toArray()
                ];
            }
        }

        return $data;
    }

    /**
     * Build comprehensive prompt for pengawas analyzing kepala
     */
    private function buildPengawasAnalysisKepalaPrompt(array $data): string
    {
        $subKomponenText = '';
        foreach ($data['sub_komponen_analysis'] as $idx => $analysis) {
            $subKomponenText .= "\n" . ($idx + 1) . ". SUB KOMPONEN: {$analysis['nama_sub_komponen']}\n";
            $subKomponenText .= "   Deskripsi: {$analysis['deskripsi']}\n";
            $subKomponenText .= "   Rata-rata Nilai Kepala (Self-Assessment): " . ($analysis['rata_rata_kepala'] ?? 'Tidak ada') . "/5.00\n";
            $subKomponenText .= "   Rata-rata Nilai Pengawas: {$analysis['rata_rata_pengawas']}/5.00\n";
            $subKomponenText .= "   Kategori: {$analysis['kategori']}\n";
            $subKomponenText .= "   Jumlah Indikator: {$analysis['jumlah_item']}\n";
            
            // Detail per indikator
            $subKomponenText .= "   Detail Indikator:\n";
            foreach ($analysis['indikator_kinerja'] as $i => $indikator) {
                $subKomponenText .= "      " . ($i + 1) . ". {$indikator['indikator']}\n";
                if (!empty($indikator['evidensi'])) {
                    $subKomponenText .= "         Evidensi: {$indikator['evidensi']}\n";
                }
                $subKomponenText .= "         Nilai Kepala: " . ($indikator['nilai_kepala'] ?? '-') . " | Nilai Pengawas: " . ($indikator['nilai_pengawas'] ?? '-') . "\n";
                if (!empty($indikator['catatan_pengawas'])) {
                    $subKomponenText .= "         Catatan Pengawas: {$indikator['catatan_pengawas']}\n";
                }
                if (!empty($indikator['bukti'])) {
                    $subKomponenText .= "         Keterangan Bukti: {$indikator['bukti']}\n";
                }
                if (!empty($indikator['url_bukti'])) {
                    $subKomponenText .= "         URL Bukti: {$indikator['url_bukti']}\n";
                }
            }
            $subKomponenText .= "\n";
        }

        $lembaga = $data['kepala']->lembagaKepala?->first();
        $namaLembaga = $lembaga?->nama_lembaga ?? 'Tidak Diketahui';

        $numSubKomponen = count($data['sub_komponen_analysis']);
        
        return "IDENTITAS ANDA: Anda adalah seorang PENGAWAS MADRASAH yang sangat berpengalaman, profesional, dan ahli dalam menilai instrumen penilaian kinerja kepala madrasah. Anda memiliki:
- Latar belakang pendidikan yang kuat dalam bidang supervisi dan evaluasi pendidikan
- Pengalaman bertahun-tahun dalam supervisi madrasah
- Kemampuan analitis yang mendalam dalam mengevaluasi kinerja kepemimpinan
- Keahlian dalam memberikan rekomendasi pembinaan yang konstruktif dan terukur

KONTEKS PENILAIAN:
Nama Kepala Madrasah: {$data['kepala']->name}
Nama Pengawas Penilai: {$data['pengawas']->name}
Nama Lembaga: {$namaLembaga}
Instrumen Penilaian: " . ($data['instrumen']->judul ?? $data['instrumen']->nama_instrumen ?? 'N/A') . "
Komponen: " . ($data['komponen']->nama_komponen ?? 'Semua Komponen') . "

RINGKASAN PENILAIAN:
- Total Indikator: {$data['jumlah_item']} item
- Indikator Terisi Kepala: {$data['jumlah_terisi_kepala']} item
- Indikator Terisi Pengawas: {$data['jumlah_terisi_pengawas']} item
- Total Nilai Kepala (Self-Assessment): {$data['total_nilai_kepala']}
- Total Nilai Pengawas: {$data['total_nilai_pengawas']}
- Rata-rata Keseluruhan Kepala: {$data['rata_rata_kepala']}/5.00
- Rata-rata Keseluruhan Pengawas: {$data['rata_rata_pengawas']}/5.00
- Jumlah Sub Komponen: {$numSubKomponen} sub komponen

ANALISIS DETAIL PER SUB KOMPONEN:
{$subKomponenText}

INSTRUKSI KHUSUS UNTUK ANALISIS LENGKAP:

1. ANALISIS PER SUB KOMPONEN (WAJIB UNTUK SETIAP SUB KOMPONEN):
   Untuk setiap dari {$numSubKomponen} sub komponen di atas, WAJIB berikan analisis detail yang mencakup:
   a. Evaluasi Pencapaian: Berikan penilaian objektif tentang pencapaian kepala dalam sub komponen tersebut berdasarkan nilai pengawas dan perbandingan dengan nilai self-assessment kepala
   b. Perbandingan Self-Assessment vs Penilaian Pengawas: Analisis gap antara nilai yang diberikan kepala dengan nilai objektif pengawas. Apakah kepala over-estimate atau under-estimate? Mengapa?
   c. Analisis Bukti dan Evidensi: Evaluasi kualitas bukti yang disajikan. Apakah bukti sesuai dengan nilai yang diberikan? Apakah ada indikator yang memerlukan bukti lebih kuat?
   d. Kekuatan Spesifik: Identifikasi minimal 2-3 kekuatan spesifik dalam sub komponen tersebut dengan menyebutkan indikator konkret
   e. Area Perbaikan: Identifikasi minimal 2-3 area yang perlu ditingkatkan dengan menyebutkan indikator konkret dan alasan mengapa perlu perbaikan
   f. Rekomendasi Konkret: Berikan rekomendasi spesifik untuk sub komponen tersebut yang dapat diimplementasikan

2. EVALUASI KUALITAS SELF-ASSESSMENT (WAJIB ANALISIS MENDALAM):
   - Analisis objektivitas kepala dalam menilai dirinya sendiri
   - Identifikasi pola: Apakah kepala cenderung over-estimate atau under-estimate di area tertentu?
   - Evaluasi konsistensi: Apakah penilaian kepala konsisten antar indikator atau ada variasi?
   - Berikan interpretasi tentang kualitas self-awareness kepala

3. IDENTIFIKASI KEKUATAN (MINIMAL 5-7 POIN):
   - List setiap kekuatan dengan menyebutkan sub komponen dan indikator spesifik
   - Jelaskan mengapa ini adalah kekuatan (dengan data pendukung)
   - Berikan contoh konkret dari data penilaian

4. IDENTIFIKASI AREA PERBAIKAN (MINIMAL 5-7 POIN):
   - List setiap area yang perlu perbaikan dengan menyebutkan sub komponen dan indikator spesifik
   - Jelaskan mengapa area ini perlu perhatian (dengan data pendukung)
   - Berikan tingkat urgensi (tinggi/sedang/rendah)

5. REKOMENDASI STRATEGIS (WAJIB DETAIL):
   a. PROGRAM PEMBINAAN: Minimal 3-4 program pembinaan spesifik dengan:
      - Nama program
      - Tujuan program
      - Sasaran (sub komponen/indikator yang ditargetkan)
      - Metode pelaksanaan
      - Durasi program
   
   b. PRIORITAS PENGEMBANGAN: Berikan prioritas dengan justifikasi:
      - Prioritas Tinggi: [list dengan alasan]
      - Prioritas Sedang: [list dengan alasan]
      - Prioritas Rendah: [list dengan alasan]
   
   c. RENCANA TINDAK LANJUT: Timeline spesifik dengan:
      - Tindakan jangka pendek (1-3 bulan)
      - Tindakan jangka menengah (3-6 bulan)
      - Tindakan jangka panjang (6-12 bulan)
      - Strategi implementasi untuk setiap tindakan
   
   d. MONITORING DAN EVALUASI: Mekanisme spesifik:
      - Frekuensi monitoring
      - Indikator keberhasilan (KPI)
      - Metode evaluasi
      - Laporan yang diperlukan

FORMAT OUTPUT (WAJIB IKUTI STRUKTUR INI DENGAN LENGKAP):

RANGKUMAN HASIL PENILAIAN KINERJA KEPALA MADRASAH

EVALUASI OBJEKTIF PENGAWAS:
Sebagai pengawas Madrasah, berikut adalah evaluasi komprehensif terhadap kinerja kepala madrasah berdasarkan instrumen penilaian yang telah dilakukan secara menyeluruh. Evaluasi ini didasarkan pada analisis mendalam terhadap {$data['jumlah_item']} indikator penilaian yang mencakup {$numSubKomponen} sub komponen utama.

ANALISIS MENDALAM PER SUB KOMPONEN:
[UNTUK SETIAP SUB KOMPONEN, WAJIB berikan analisis lengkap yang mencakup semua poin di bawah ini]

Sub Komponen 1: [Nama Sub Komponen]
Evaluasi Pencapaian: [Analisis lengkap berdasarkan nilai pengawas dan perbandingan dengan nilai kepala]
Perbandingan Self-Assessment: [Analisis gap antara nilai kepala dan pengawas, interpretasi objektivitas]
Analisis Bukti dan Evidensi: [Evaluasi kualitas bukti yang disajikan]
Kekuatan Spesifik:
- [Kekuatan 1 dengan indikator spesifik]
- [Kekuatan 2 dengan indikator spesifik]
- [Kekuatan 3 dengan indikator spesifik]
Area Perbaikan:
- [Area 1 dengan indikator spesifik dan alasan]
- [Area 2 dengan indikator spesifik dan alasan]
Rekomendasi untuk Sub Komponen ini: [Rekomendasi spesifik dan terukur]

[ULANGI UNTUK SETIAP SUB KOMPONEN LAINNYA - TOTAL {$numSubKomponen} SUB KOMPONEN HARUS DIANALISIS]

RINGKASAN KESELURUHAN:
Total Indikator Dinilai: {$data['jumlah_item']} item
Rata-rata Keseluruhan Kepala (Self-Assessment): {$data['rata_rata_kepala']}/5.00
Rata-rata Keseluruhan Pengawas: {$data['rata_rata_pengawas']}/5.00
Kategori Penilaian Keseluruhan: [Sangat Baik/Baik/Cukup/Kurang]
Gap Analysis: [Hitung selisih antara rata-rata kepala dan pengawas, berikan interpretasi mendalam tentang gap ini. Apakah kepala cenderung over-estimate atau under-estimate? Apa implikasinya?]

KEKUATAN YANG DITEMUKAN:
[List minimal 5-7 kekuatan yang menonjol dengan spesifikasi detail]
1. [Nama Kekuatan] - Sub Komponen: [nama], Indikator: [indikator spesifik]
   Penjelasan: [Penjelasan lengkap mengapa ini adalah kekuatan, dengan data pendukung]
2. [Nama Kekuatan] - Sub Komponen: [nama], Indikator: [indikator spesifik]
   Penjelasan: [Penjelasan lengkap]
[Lanjutkan untuk setiap kekuatan]

AREA YANG MEMERLUKAN PERHATIAN KHUSUS:
[List minimal 5-7 area yang perlu perbaikan dengan spesifikasi detail]
1. [Nama Area] - Sub Komponen: [nama], Indikator: [indikator spesifik], Urgensi: [Tinggi/Sedang/Rendah]
   Penjelasan: [Penjelasan lengkap mengapa area ini perlu perhatian, dengan data pendukung]
   Dampak: [Jelaskan dampak jika tidak diperbaiki]
2. [Nama Area] - Sub Komponen: [nama], Indikator: [indikator spesifik], Urgensi: [Tinggi/Sedang/Rendah]
   Penjelasan: [Penjelasan lengkap]
[Lanjutkan untuk setiap area]

EVALUASI KUALITAS SELF-ASSESSMENT:
[Analisis objektif dan mendalam tentang kualitas penilaian diri yang dilakukan kepala]
- Tingkat Objektivitas: [Apakah kepala sudah objektif dalam menilai dirinya sendiri? Berikan analisis dengan contoh konkret dari data]
- Pola Penilaian: [Apakah ada indikasi over-estimation atau under-estimation? Di sub komponen mana? Mengapa?]
- Konsistensi Penilaian: [Bagaimana konsistensi penilaian kepala terhadap berbagai indikator? Apakah ada pola tertentu?]
- Self-Awareness: [Evaluasi tingkat self-awareness kepala berdasarkan perbandingan nilai self-assessment dengan nilai objektif pengawas]
- Implikasi: [Apa implikasi dari pola penilaian kepala terhadap pengembangan profesionalnya?]

REKOMENDASI STRATEGIS PENGAWAS:

1. PROGRAM PEMBINAAN:
   Program 1: [Nama Program]
   Tujuan: [Tujuan spesifik]
   Sasaran: [Sub komponen/indikator yang ditargetkan]
   Metode: [Metode pelaksanaan yang detail]
   Durasi: [Durasi program]
   Indikator Keberhasilan: [KPI yang terukur]
   
   Program 2: [Nama Program]
   [Detail lengkap seperti Program 1]
   
   Program 3: [Nama Program]
   [Detail lengkap seperti Program 1]
   
   Program 4: [Nama Program]
   [Detail lengkap seperti Program 1]

2. PRIORITAS PENGEMBANGAN:
   Prioritas Tinggi:
   - [Area 1]: [Alasan mengapa prioritas tinggi]
   - [Area 2]: [Alasan mengapa prioritas tinggi]
   - [Area 3]: [Alasan mengapa prioritas tinggi]
   
   Prioritas Sedang:
   - [Area 1]: [Alasan mengapa prioritas sedang]
   - [Area 2]: [Alasan mengapa prioritas sedang]
   
   Prioritas Rendah:
   - [Area 1]: [Alasan mengapa prioritas rendah]

3. RENCANA TINDAK LANJUT:
   Tindakan Jangka Pendek (1-3 bulan):
   - [Tindakan 1]: [Detail implementasi, timeline, responsible person]
   - [Tindakan 2]: [Detail implementasi, timeline, responsible person]
   - [Tindakan 3]: [Detail implementasi, timeline, responsible person]
   
   Tindakan Jangka Menengah (3-6 bulan):
   - [Tindakan 1]: [Detail implementasi, timeline, responsible person]
   - [Tindakan 2]: [Detail implementasi, timeline, responsible person]
   
   Tindakan Jangka Panjang (6-12 bulan):
   - [Tindakan 1]: [Detail implementasi, timeline, responsible person]
   - [Tindakan 2]: [Detail implementasi, timeline, responsible person]

4. MONITORING DAN EVALUASI:
   Frekuensi Monitoring: [Berapa kali per bulan/kuartal]
   Indikator Keberhasilan (KPI): [List KPI yang terukur]
   Metode Evaluasi: [Metode yang akan digunakan]
   Format Laporan: [Format laporan yang diperlukan]
   Mekanisme Feedback: [Bagaimana feedback akan diberikan]

KESIMPULAN DAN ARAHAN PENGAWAS:
[Kesimpulan profesional yang memberikan arahan jelas, konstruktif, dan mendukung pengembangan kepala madrasah. Harus mencakup:
- Ringkasan temuan utama
- Arahan strategis untuk kepala
- Harapan pengawas terhadap perbaikan
- Komitmen pengawas untuk mendukung pengembangan]

CATATAN PENTING UNTUK OUTPUT:
- PANJANG MINIMUM: Analisis HARUS minimal 1200-1500 kata untuk memastikan kelengkapan
- DETAIL: Setiap sub komponen HARUS dianalisis secara detail dengan minimal 150-200 kata per sub komponen
- SPESIFIK: Semua rekomendasi HARUS spesifik, terukur, dan dapat diimplementasikan
- BERBASIS DATA: Semua analisis HARUS mengacu pada data konkret dari penilaian
- PROFESIONAL: Gunakan bahasa profesional namun mudah dipahami
- KONSTRUKTIF: Fokus pada pengembangan, bukan hanya kritik

TONALITAS: Profesional, objektif, analitis, konstruktif, supportive, berbasis data, memberikan arahan yang jelas dan terukur, menunjukkan pengalaman dan keahlian sebagai pengawas profesional.

PENTING MUTLAK: 
- Jangan gunakan markdown formatting seperti ** atau * dalam output
- Gunakan format text biasa saja
- Setiap sub komponen ({$numSubKomponen} sub komponen) HARUS dianalisis secara detail
- Analisis HARUS mencakup semua bagian yang diminta di atas
- Minimum 1200-1500 kata untuk memastikan analisis yang benar-benar komprehensif
- Jangan memberikan ringkasan singkat - berikan analisis lengkap dan mendalam";
    }

    /**
     * Generate manual analysis when AI fails (for kepala)
     */
    private function generateManualAnalysisKepala(\App\Models\PenilaianKepala $penilaian): string
    {
        $jawaban = $penilaian->jawabanKepala;
        $totalNilaiPengawas = $jawaban->sum('nilai_pengawas');
        $totalNilaiKepala = $jawaban->sum('nilai_guru_self');
        $jumlahItemTerisiPengawas = $jawaban->whereNotNull('nilai_pengawas')->where('nilai_pengawas', '>', 0)->count();
        $jumlahItemTerisiKepala = $jawaban->whereNotNull('nilai_guru_self')->where('nilai_guru_self', '>', 0)->count();
        $rataRataPengawas = $jumlahItemTerisiPengawas > 0 ? round($totalNilaiPengawas / $jumlahItemTerisiPengawas, 2) : 0;
        $rataRataKepala = $jumlahItemTerisiKepala > 0 ? round($totalNilaiKepala / $jumlahItemTerisiKepala, 2) : 0;
        $kategori = $this->determineCategory($rataRataPengawas);

        $text = "RINGKASAN PENILAIAN KINERJA KEPALA MADRASAH\n\n";
        $text .= "Nama Kepala: {$penilaian->kepala->name}\n";
        $text .= "Total indikator dinilai: {$jawaban->count()} item\n";
        $text .= "Indikator terisi kepala: {$jumlahItemTerisiKepala} item\n";
        $text .= "Indikator terisi pengawas: {$jumlahItemTerisiPengawas} item\n";
        $text .= "Rata-rata nilai kepala (self-assessment): {$rataRataKepala}/5.00\n";
        $text .= "Rata-rata nilai pengawas: {$rataRataPengawas}/5.00 (Kategori: {$kategori})\n\n";
        
        $text .= "Rekomendasi umum: ";
        if ($rataRataPengawas >= 4.5) {
            $text .= "Kepala madrasah menunjukkan kinerja yang sangat baik. Pertahankan kinerja ini dan lakukan berbagi praktik baik serta program mentoring internal untuk meningkatkan kapasitas organisasi secara keseluruhan.";
        } elseif ($rataRataPengawas >= 3.5) {
            $text .= "Kepala madrasah menunjukkan kinerja yang baik. Fokuskan perbaikan pada indikator yang masih dalam kategori sedang dengan menyusun rencana aksi yang terukur dan timeline yang jelas.";
        } elseif ($rataRataPengawas >= 2.5) {
            $text .= "Kepala madrasah menunjukkan kinerja yang cukup. Diperlukan pendampingan dan supervisi berkala pada indikator yang bernilai rendah, serta program pembinaan yang intensif.";
        } else {
            $text .= "Kepala madrasah memerlukan perhatian khusus. Susun program pembinaan intensif dengan monitoring ketat dan evaluasi berkala untuk meningkatkan kinerja.";
        }

        return $text;
    }
}
