<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Validation\Rule;

class Lembaga extends Model
{
    use HasFactory;

    protected $table = 'lembaga';

    protected $fillable = [
        'pengawas_id',
        'kepala_id',
        'npsn_nsm',
        'nama_lembaga',
        'jenis',
        'alamat_lengkap',
        'kelurahan',
        'kecamatan',
        'kabupaten',
        'provinsi',
        'kode_pos',
        'telepon',
        'email',
        'website',
        'nama_kepala',
        'nip_kepala',
        'akreditasi',
        'tahun_berdiri',
        'logo_lembaga',
    ];

    protected $casts = [
        'tahun_berdiri' => 'integer',
    ];

    // Static property to cache regions data
    private static $regionsData = null;

    // Helper method to get regions data
    private static function getRegionsData()
    {
        if (self::$regionsData === null) {
            $jsonPath = storage_path('app/indonesia-regions.json');
            if (file_exists($jsonPath)) {
                self::$regionsData = json_decode(file_get_contents($jsonPath), true);
            } else {
                self::$regionsData = ['provinces' => [], 'regencies' => [], 'districts' => [], 'villages' => []];
            }
        }
        return self::$regionsData;
    }

    // Accessors untuk mengkonversi kode ID menjadi nama wilayah
    public function getProvinsiNameAttribute()
    {
        if (!$this->provinsi) return null;
        
        $regionsData = self::getRegionsData();
        $provinsi = collect($regionsData['provinces'])->firstWhere('id', $this->provinsi);
        return $provinsi ? $provinsi['name'] : $this->provinsi;
    }

    public function getKabupatenNameAttribute()
    {
        if (!$this->kabupaten) return null;
        
        $regionsData = self::getRegionsData();
        $regencies = $regionsData['regencies'][$this->provinsi] ?? [];
        $kabupaten = collect($regencies)->firstWhere('id', $this->kabupaten);
        return $kabupaten ? $kabupaten['name'] : $this->kabupaten;
    }

    public function getKecamatanNameAttribute()
    {
        if (!$this->kecamatan) return null;
        
        $regionsData = self::getRegionsData();
        $districts = $regionsData['districts'][$this->kabupaten] ?? [];
        $kecamatan = collect($districts)->firstWhere('id', $this->kecamatan);
        return $kecamatan ? $kecamatan['name'] : $this->kecamatan;
    }

    public function getKelurahanNameAttribute()
    {
        if (!$this->kelurahan) return null;
        
        $regionsData = self::getRegionsData();
        $villages = $regionsData['villages'][$this->kecamatan] ?? [];
        $kelurahan = collect($villages)->firstWhere('id', $this->kelurahan);
        return $kelurahan ? $kelurahan['name'] : $this->kelurahan;
    }

    // Relationships
    public function pengawas()
    {
        return $this->belongsTo(User::class, 'pengawas_id');
    }

    public function kepala()
    {
        return $this->belongsTo(User::class, 'kepala_id');
    }

    public function guru()
    {
        return $this->hasMany(Guru::class);
    }

    public function distribusiInstrumen()
    {
        return $this->hasMany(DistribusiInstrumen::class);
    }

    // Validation rules untuk admin create (hanya 3 field wajib)
    public static function validationRulesAdmin($id = null)
    {
        return [
            'npsn_nsm' => [
                'required',
                'string',
                'max:20',
                Rule::unique('lembaga')->ignore($id),
            ],
            'nama_lembaga' => 'required|string|max:255',
            'jenis' => 'required|in:RA,MI,MTs,MA,ra,mi,mts,ma,Ra,Mi,Mts,Ma',
        ];
    }

    // Validation rules untuk pengawas/kepala (lengkap)
    public static function validationRulesComplete($id = null)
    {
        return [
            'pengawas_id' => 'nullable|exists:users,id',
            'npsn_nsm' => [
                'required',
                'string',
                'max:20',
                Rule::unique('lembaga')->ignore($id),
            ],
            'nama_lembaga' => 'required|string|max:255',
            'jenis' => 'required|in:RA,MI,MTs,MA,ra,mi,mts,ma,Ra,Mi,Mts,Ma',
            'alamat_lengkap' => 'required|string',
            'kelurahan' => 'required|string|max:100',
            'kecamatan' => 'required|string|max:100',
            'kabupaten' => 'required|string|max:100',
            'provinsi' => 'required|string|max:100',
            'kode_pos' => 'nullable|string|max:10',
            'telepon' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|url|max:255',
            // 'nama_kepala' => 'required|string|max:255', // Removed - auto-synced from kepala relationship
            'nip_kepala' => 'required|string|max:30',
            'akreditasi' => 'nullable|string|max:10',
            'tahun_berdiri' => 'nullable|integer|min:1900|max:' . date('Y'),
        ];
    }

    // Validation rules untuk admin edit (regional fields optional, kepala fields read-only)
    public static function validationRulesAdminEdit($id = null)
    {
        return [
            'pengawas_id' => 'nullable|exists:users,id',
            'npsn_nsm' => [
                'required',
                'string',
                'max:20',
                Rule::unique('lembaga')->ignore($id),
            ],
            'nama_lembaga' => 'required|string|max:255',
            'jenis' => 'required|in:RA,MI,MTs,MA,ra,mi,mts,ma,Ra,Mi,Mts,Ma',
            'alamat_lengkap' => 'required|string',
            'kelurahan' => 'nullable|string|max:100',
            'kecamatan' => 'nullable|string|max:100',
            'kabupaten' => 'nullable|string|max:100',
            'provinsi' => 'nullable|string|max:100',
            'kode_pos' => 'nullable|string|max:10',
            'telepon' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|url|max:255',
            'nama_kepala' => 'nullable|string|max:255', // Read-only field, validation not enforced
            'nip_kepala' => 'nullable|string|max:30', // Read-only field, validation not enforced
            'akreditasi' => 'nullable|string|max:10',
            'tahun_berdiri' => 'nullable|integer|min:1900|max:' . date('Y'),
        ];
    }

    // Backward compatibility
    public static function validationRules($id = null)
    {
        return self::validationRulesComplete($id);
    }
}
