<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Guru extends Model
{
    use HasFactory;

    protected $table = 'guru';

    protected $fillable = [
        'user_id',
        'lembaga_id',
        'nip',
        'nuptk',
        'nama_lengkap',
        'gelar_depan',
        'gelar_belakang',
        'tempat_lahir',
        'tanggal_lahir',
        'jenis_kelamin',
        'pendidikan_terakhir',
        'jurusan',
        'mata_pelajaran',
        'kelas_diampu',
        'status_kepegawaian',
        'status_sertifikasi',
        'tahun_sertifikasi',
        'no_sertifikat',
        'status_inpassing',
        'golongan',
        'tmt_golongan',
        'alamat',
        'telepon',
    ];

    protected $casts = [
        'tanggal_lahir' => 'date',
        'tahun_sertifikasi' => 'integer',
        'tmt_golongan' => 'date',
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function lembaga()
    {
        return $this->belongsTo(Lembaga::class);
    }

    public function penilaianGuru()
    {
        return $this->hasMany(PenilaianGuru::class);
    }

    // Accessor untuk nama lengkap dengan gelar
    public function getNamaLengkapWithGelarAttribute()
    {
        $gelarDepan = $this->gelar_depan ? $this->gelar_depan . ' ' : '';
        $gelarBelakang = $this->gelar_belakang ? ', ' . $this->gelar_belakang : '';
        
        return $gelarDepan . $this->nama_lengkap . $gelarBelakang;
    }

    // Validation rules
    public static function validationRules($id = null)
    {
        return [
            'user_id' => 'required|exists:users,id',
            'lembaga_id' => 'required|exists:lembaga,id',
            'nip' => 'nullable|string|max:30',
            'nuptk' => 'nullable|string|max:30',
            'nama_lengkap' => 'required|string|max:255',
            'gelar_depan' => 'nullable|string|max:50',
            'gelar_belakang' => 'nullable|string|max:50',
            'tempat_lahir' => 'required|string|max:100',
            'tanggal_lahir' => 'required|date|before:today',
            'jenis_kelamin' => 'required|in:L,P',
            'pendidikan_terakhir' => 'required|string|max:100',
            'jurusan' => 'nullable|string|max:100',
            'mata_pelajaran' => 'required|string|max:100',
            'kelas_diampu' => 'required|string|max:100',
            'status_kepegawaian' => 'required|in:PNS,PPPK,GTY,GTT',
            'status_sertifikasi' => 'required|in:sudah,belum',
            'tahun_sertifikasi' => 'nullable|integer|min:1990|max:' . date('Y'),
            'no_sertifikat' => 'nullable|string|max:100',
            'status_inpassing' => 'required|in:sudah,belum',
            'golongan' => 'nullable|string|max:20',
            'tmt_golongan' => 'nullable|date',
            'alamat' => 'required|string',
            'telepon' => 'nullable|string|max:20',
        ];
    }
}
