<?php

namespace App\Imports;

use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\Importable;
use Spatie\Permission\Models\Role;

class UsersImport implements ToCollection, WithHeadingRow, WithValidation, WithChunkReading
{
    use Importable;

    protected $importedCount = 0;
    protected $errors = [];

    /**
     * @param Collection $collection
     */
    public function collection(Collection $collection)
    {
        foreach ($collection as $row) {
            try {
                // Skip if email already exists
                if (User::where('email', $row['email'])->exists()) {
                    $this->errors[] = "Email {$row['email']} sudah ada, dilewati.";
                    continue;
                }

                // Create user
                $user = User::create([
                    'name' => $row['nama_lengkap'],
                    'email' => $row['email'],
                    'password' => Hash::make($row['password'] ?? 'password123'),
                    'phone' => $row['phone'] ?? null,
                    'is_active' => $this->parseStatus($row['status'] ?? 'Aktif'),
                ]);

                // Assign role if specified
                if (isset($row['role']) && $row['role']) {
                    $role = Role::where('name', strtolower($row['role']))->first();
                    if ($role) {
                        $user->assignRole($role);
                    }
                }

                $this->importedCount++;
            } catch (\Exception $e) {
                $this->errors[] = "Error pada baris {$row['nama_lengkap']}: " . $e->getMessage();
            }
        }
    }

    /**
     * @return array
     */
    public function rules(): array
    {
        return [
            '*.nama_lengkap' => 'required|string|max:255',
            '*.email' => 'required|email|max:255',
            '*.password' => 'nullable|string|min:6',
            '*.phone' => 'nullable|string|max:20',
            '*.role' => 'nullable|string|in:admin,pengawas,kepala,guru,Admin,Pengawas,Kepala,Guru,ADMIN,PENGAWAS,KEPALA,GURU',
            '*.status' => 'nullable|string|in:Aktif,Nonaktif,aktif,nonaktif,AKTIF,NONAKTIF',
        ];
    }

    /**
     * @return array
     */
    public function customValidationMessages()
    {
        return [
            '*.nama_lengkap.required' => 'Nama lengkap harus diisi.',
            '*.email.required' => 'Email harus diisi.',
            '*.email.email' => 'Format email tidak valid.',
            '*.password.min' => 'Password minimal 6 karakter.',
            '*.role.in' => 'Role harus salah satu dari: admin, pengawas, kepala, guru (case-insensitive).',
            '*.status.in' => 'Status harus salah satu dari: Aktif, Nonaktif (case-insensitive).',
        ];
    }

    /**
     * @return int
     */
    public function chunkSize(): int
    {
        return 100;
    }

    /**
     * Parse status string to boolean
     */
    private function parseStatus($status)
    {
        return strtolower($status) === 'aktif';
    }

    /**
     * Get import results
     */
    public function getResults()
    {
        return [
            'imported_count' => $this->importedCount,
            'errors' => $this->errors
        ];
    }
}
