<?php

namespace App\Http\Controllers\Pengawas;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\PenilaianKepala;
use App\Models\ValidasiPengawas;
use App\Models\Lembaga;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ValidasiController extends Controller
{
    /**
     * Display a listing of penilaian that need validation
     */
    public function index()
    {
        $user = Auth::user();
        
        // Get lembaga binaan by this pengawas
        $lembagaIds = Lembaga::where('pengawas_id', $user->id)->pluck('id');
        
        // Get penilaian kepala yang disubmit dan menunggu validasi (opsional)
        $penilaianKepala = PenilaianKepala::with([
            'kepala',
            'distribusiInstrumen.instrumen',
            'distribusiInstrumen.komponen',
            'validasiPengawas'
        ])
        ->where('pengawas_id', $user->id)
        ->where('status', 'submitted')
        ->orderBy('updated_at', 'desc')
        ->paginate(10);

        // Get statistics
        $stats = [
            'total_pending' => PenilaianKepala::where('pengawas_id', $user->id)->where('status', 'submitted')->count(),
            'total_validated' => ValidasiPengawas::where('pengawas_id', $user->id)->count(),
            'approved' => ValidasiPengawas::where('pengawas_id', $user->id)->where('status', 'approved')->count(),
            'revision' => ValidasiPengawas::where('pengawas_id', $user->id)->where('status', 'revision')->count(),
        ];

        return view('pengawas.validasi.index', ['penilaianGuru' => $penilaianKepala, 'stats' => $stats]);
    }

    /**
     * Show the form for validating a specific penilaian
     */
    public function show($id)
    {
        $user = Auth::user();
        
        // Get lembaga binaan by this pengawas
        $lembagaIds = Lembaga::where('pengawas_id', $user->id)->pluck('id');
        
        // Get penilaian kepala yang perlu validasi
        $penilaianGuru = PenilaianKepala::with([
            'kepala',
            'distribusiInstrumen.instrumen',
            'distribusiInstrumen.komponen',
            'jawabanKepala.instrumenItem',
            'validasiPengawas'
        ])
        ->where('pengawas_id', $user->id)
        ->where('status', 'submitted')
        ->findOrFail($id);

        // Get jawaban guru with instrumen items
        $jawabanGuru = $penilaianGuru->jawabanKepala()
            ->leftJoin('instrumen_penilaians', 'jawaban_kepala.instrumen_item_id', '=', 'instrumen_penilaians.id')
            ->with('instrumenItem')
            ->orderBy('instrumen_penilaians.no_urutan', 'asc')
            ->select('jawaban_kepala.*')
            ->get();

        return view('pengawas.validasi.show', compact('penilaianGuru', 'jawabanGuru'));
    }

    /**
     * Store validation result
     */
    public function store(Request $request, $id)
    {
        $user = Auth::user();
        
        // Validate request
        $request->validate([
            'status' => 'required|in:approved,revision',
            'feedback' => 'nullable|string|max:1000',
        ]);

        // Get lembaga binaan by this pengawas
        $lembagaIds = Lembaga::where('pengawas_id', $user->id)->pluck('id');
        
        // Get penilaian kepala
        $penilaianGuru = PenilaianKepala::where('pengawas_id', $user->id)
            ->where('status', 'submitted')
            ->findOrFail($id);

        try {
            DB::beginTransaction();

            // Create or update validasi pengawas
            ValidasiPengawas::updateOrCreate([
                'penilaian_kepala_id' => $penilaianGuru->id,
                'pengawas_id' => $user->id,
            ], [
                'status' => $request->status,
                'feedback' => $request->feedback,
                'tanggal_validasi' => now(),
            ]);

            // Update penilaian guru status based on validation result
            $penilaianGuru->update(['status' => $request->status === 'approved' ? 'final' : 'revision']);

            DB::commit();

            return redirect()->route('pengawas.validasi.index')
                ->with('success', 'Validasi berhasil disimpan!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()
                ->with('error', 'Terjadi kesalahan saat menyimpan validasi: ' . $e->getMessage());
        }
    }

    /**
     * Get validation statistics
     */
    public function getStats()
    {
        $user = Auth::user();
        
        // Get lembaga binaan by this pengawas
        $lembagaIds = Lembaga::where('pengawas_id', $user->id)->pluck('id');
        
        $stats = [
            'pending' => PenilaianKepala::where('pengawas_id', $user->id)->where('status', 'submitted')->count(),
            'approved' => ValidasiPengawas::where('pengawas_id', $user->id)->where('status', 'approved')->count(),
            'revision' => ValidasiPengawas::where('pengawas_id', $user->id)->where('status', 'revision')->count(),
        ];

        return response()->json($stats);
    }
}
