<?php

namespace App\Http\Controllers\Pengawas;

use App\Http\Controllers\Controller;
use App\Models\PenilaianKepala;
use App\Services\AIAnalysisService;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Auth;

class RaportController extends Controller
{
    public function generateRaport($id)
    {
        $user = Auth::user();

        $penilaian = PenilaianKepala::with([
            'kepala.lembagaKepala',
            'pengawas',
            'distribusiInstrumen.instrumen',
            'distribusiInstrumen.komponen',
            'jawabanKepala.instrumenItem.subKomponen.komponenPenilaian',
        ])
            ->where('pengawas_id', $user->id)
            ->findOrFail($id);

        if (!in_array($penilaian->status, ['validated', 'final'])) {
            return redirect()->back()->with('error', 'Raport hanya untuk penilaian berstatus validated atau final.');
        }

        $jawaban = $penilaian->jawabanKepala()
            ->leftJoin('instrumen_penilaians', 'jawaban_kepala.instrumen_item_id', '=', 'instrumen_penilaians.id')
            ->leftJoin('sub_komponens', 'instrumen_penilaians.sub_komponen_id', '=', 'sub_komponens.id')
            ->leftJoin('komponen_penilaians', 'sub_komponens.komponen_penilaian_id', '=', 'komponen_penilaians.id')
            ->with('instrumenItem')
            ->orderBy('komponen_penilaians.urutan', 'asc')
            ->orderBy('sub_komponens.urutan', 'asc')
            ->orderBy('instrumen_penilaians.no_urutan', 'asc')
            ->select('jawaban_kepala.*')
            ->get();

        // Group jawaban by sub komponen for analysis
        $subKomponenAnalysis = $this->prepareSubKomponenAnalysis($jawaban);

        // Load AI analysis from database (sudah di-generate saat finalisasi)
        // Jika belum ada, hanya generate sebagai fallback untuk backward compatibility
        // Tapi seharusnya sudah ada karena di-generate saat finalisasi
        $analisisPengawas = $penilaian->catatan_pengawas;
        
        // Fallback: Generate hanya jika benar-benar belum ada atau error
        // (seharusnya sudah di-generate saat finalisasi)
        if (!$analisisPengawas || 
            strpos($analisisPengawas, 'Terjadi kesalahan') !== false ||
            strpos($analisisPengawas, 'Analisis AI tidak diaktifkan') !== false ||
            strpos($analisisPengawas, 'Analisis tidak dapat dihasilkan') !== false) {
            
            \Log::warning('AI analysis not found in database, generating as fallback', [
                'penilaian_id' => $penilaian->id,
                'status' => $penilaian->status
            ]);
            
            try {
                $aiService = new AIAnalysisService();
                if ($aiService->isEnabled()) {
                    $analisisPengawas = $aiService->generatePengawasAnalysisKepala($penilaian);
                    // Update the database with new analysis
                    $penilaian->update(['catatan_pengawas' => $analisisPengawas]);
                } else {
                    $analisisPengawas = 'Analisis AI tidak diaktifkan. Silakan hubungi administrator untuk mengaktifkan fitur AI.';
                }
            } catch (\Exception $e) {
                \Log::error('Pengawas AI Analysis Error (fallback): ' . $e->getMessage());
                $analisisPengawas = 'Terjadi kesalahan dalam menghasilkan analisis. Silakan finalisasi ulang penilaian untuk generate analisis AI.';
            }
        }

        // Calculate total and average
        $totalNilai = $jawaban->sum('nilai_pengawas');
        $jumlahItem = $jawaban->whereNotNull('nilai_pengawas')->where('nilai_pengawas', '>', 0)->count();
        $rataRata = $jumlahItem > 0 ? round($totalNilai / $jumlahItem, 2) : 0;

        // Get lembaga - lembagaKepala() returns collection, so use first()
        $lembaga = $penilaian->kepala?->lembagaKepala?->first();
        
        // Debug: Log lembaga and logo info
        if ($lembaga) {
            \Log::info('Raport Lembaga Info:', [
                'lembaga_id' => $lembaga->id,
                'lembaga_nama' => $lembaga->nama_lembaga,
                'logo_lembaga' => $lembaga->logo_lembaga,
                'logo_path_exists' => $lembaga->logo_lembaga ? file_exists(storage_path('app/public/' . $lembaga->logo_lembaga)) : false,
            ]);
        }

        // Get kabupaten from lembaga
        $kabupaten = '';
        if ($lembaga) {
            // Use accessor kabupaten_name (which uses getKabupatenNameAttribute)
            // If accessor fails, fallback to direct property kabupaten
            try {
                $kabupaten = $lembaga->kabupaten_name;
            } catch (\Exception $e) {
                $kabupaten = $lembaga->kabupaten ?? '';
            }
            
            // If still empty, try direct property
            if (empty($kabupaten)) {
                $kabupaten = $lembaga->kabupaten ?? '';
            }
        }

        $data = [
            'penilaian' => $penilaian,
            'jawaban' => $jawaban,
            'subKomponenAnalysis' => $subKomponenAnalysis,
            'analisisPengawas' => $analisisPengawas,
            'kepala' => $penilaian->kepala,
            'pengawas' => $penilaian->pengawas,
            'lembaga' => $lembaga,
            'instrumen' => $penilaian->distribusiInstrumen?->instrumen,
            'komponen' => $penilaian->distribusiInstrumen?->komponen,
            'totalNilai' => $totalNilai,
            'rataRata' => $rataRata,
            'tanggal' => now()->format('d F Y'),
            'tahun' => now()->year,
            'kabupaten' => $this->formatKabupaten($kabupaten),
        ];

        $pdf = Pdf::loadView('pengawas.raport.template', $data);
        $pdf->setPaper('A4', 'portrait');

        $filename = 'Raport_Penilaian_Kepala_' . ($penilaian->kepala?->name ?? 'kepala') . '_' . now()->format('Y-m-d') . '.pdf';
        return $pdf->download($filename);
    }

    /**
     * Prepare sub komponen analysis data
     */
    private function prepareSubKomponenAnalysis($jawaban)
    {
        // Group by sub_komponen_id using callback to ensure proper grouping
        $groupedBySubKomponen = $jawaban->groupBy(function($jawaban) {
            return $jawaban->instrumenItem?->sub_komponen_id ?? 'unknown';
        });
        
        $analysis = [];

        foreach ($groupedBySubKomponen as $subKomponenId => $jawabanGroup) {
            // Skip if sub_komponen_id is unknown or first item doesn't have instrumenItem
            if ($subKomponenId === 'unknown' || !$jawabanGroup->first()->instrumenItem) {
                continue;
            }
            
            $subKomponen = $jawabanGroup->first()->instrumenItem->subKomponen;
            
            // Filter nilai_pengawas yang valid (not null and > 0)
            $nilaiPengawas = $jawabanGroup->pluck('nilai_pengawas')->filter(function($v) {
                return $v !== null && $v !== '' && $v > 0;
            })->values();
            
            // Filter nilai_kepala (nilai_guru_self) yang valid (not null and > 0)
            $nilaiKepala = $jawabanGroup->pluck('nilai_guru_self')->filter(function($v) {
                return $v !== null && $v !== '' && $v > 0;
            })->values();
            
            // Calculate rata-rata hanya jika ada nilai pengawas yang valid
            if ($nilaiPengawas->isNotEmpty()) {
                $rataRataPengawas = $nilaiPengawas->avg();
                $kategori = $this->determineCategory($rataRataPengawas);
                
                // Calculate rata-rata kepala jika ada nilai yang valid
                $rataRataKepala = null;
                if ($nilaiKepala->isNotEmpty()) {
                    $rataRataKepala = $nilaiKepala->avg();
                }
                
                $analysis[] = [
                    'nama_sub_komponen' => $subKomponen->nama_sub_komponen ?? 'Unknown',
                    'deskripsi' => $subKomponen->deskripsi ?? '',
                    'rata_rata_pengawas' => round($rataRataPengawas, 2),
                    'rata_rata_kepala' => $rataRataKepala ? round($rataRataKepala, 2) : null,
                    'kategori' => $kategori,
                    'jumlah_item' => $jawabanGroup->count(),
                    'jumlah_terisi_pengawas' => $nilaiPengawas->count(),
                    'jumlah_terisi_kepala' => $nilaiKepala->count(),
                    'nilai_detail' => $nilaiPengawas->toArray(),
                    'indikator_kinerja' => $jawabanGroup->map(function($jawaban) {
                        return [
                            'indikator' => $jawaban->instrumenItem?->indikator_kinerja ?? 'Unknown',
                            'nilai_kepala' => $jawaban->nilai_guru_self ?? '-',
                            'nilai_pengawas' => $jawaban->nilai_pengawas ?? '-',
                            'bukti' => $jawaban->keterangan_bukti ?? '-',
                            'url_bukti' => $jawaban->url_bukti ?? '-',
                        ];
                    })->toArray()
                ];
            }
        }

        // Sort by sub_komponen urutan if available
        usort($analysis, function($a, $b) {
            // Try to maintain order from database
            return 0;
        });

        return $analysis;
    }

    /**
     * Determine category based on average score
     */
    private function determineCategory($rataRata)
    {
        if ($rataRata >= 4.5) {
            return 'Sangat Baik';
        } elseif ($rataRata >= 3.5) {
            return 'Baik';
        } elseif ($rataRata >= 2.5) {
            return 'Cukup';
        } elseif ($rataRata >= 1.5) {
            return 'Kurang';
        } else {
            return 'Sangat Kurang';
        }
    }

    /**
     * Format kabupaten name
     */
    private function formatKabupaten($kabupaten)
    {
        if (empty($kabupaten)) {
            return '';
        }
        
        // Remove common prefixes
        $kabupaten = preg_replace('/^(KABUPATEN|KOTA|KAB\.|KOTA\.)\s*/i', '', $kabupaten);
        
        return ucwords(strtolower($kabupaten));
    }
}


