<?php

namespace App\Http\Controllers\Pengawas;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\PenilaianKepala;
use App\Models\JawabanKepala;
use App\Models\DistribusiInstrumen;
use App\Models\InstrumenPenilaian;
use App\Models\Lembaga;
use App\Services\AIAnalysisService;

class PenilaianKepalaController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        
        // Build query with relationships
        $query = PenilaianKepala::with(['kepala', 'distribusiInstrumen.lembaga'])
            ->where('pengawas_id', $user->id);
        
        // Apply filters
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        if ($request->filled('lembaga_id')) {
            $query->whereHas('distribusiInstrumen', function($q) use ($request) {
                $q->where('lembaga_id', $request->lembaga_id);
            });
        }
        
        // Order by created_at desc
        $items = $query->orderBy('created_at', 'desc')->paginate(10);
        
        // Get list of lembaga for filter dropdown
        $lembagaList = Lembaga::where('pengawas_id', $user->id)
            ->orderBy('nama_lembaga')
            ->get();
        
        // Get current filter values
        $filters = [
            'status' => $request->status,
            'lembaga_id' => $request->lembaga_id,
        ];

        return view('pengawas.penilaian.index', compact('items', 'lembagaList', 'filters'));
    }

    public function show($id)
    {
        $user = Auth::user();
        $penilaian = PenilaianKepala::with(['jawabanKepala', 'kepala', 'distribusiInstrumen'])
            ->where('pengawas_id', $user->id)
            ->findOrFail($id);

        $distribusi = $penilaian->distribusiInstrumen;
        if ($distribusi->komponen) {
            // Fix: Add secondary sort by id to ensure consistent ordering when no_urutan is duplicate
            $itemsQuery = $distribusi->komponen->instrumenPenilaians()->orderBy('no_urutan')->orderBy('id');
        } else {
            $subIds = $distribusi->instrumen->komponenPenilaians->flatMap(function ($k) {
                return $k->subKomponens->pluck('id');
            })->values();
            // Fix: Add secondary sort by id to ensure consistent ordering when no_urutan is duplicate
            $itemsQuery = InstrumenPenilaian::whereIn('sub_komponen_id', $subIds)->orderBy('no_urutan')->orderBy('id');
        }
        $totalItems = $itemsQuery->count();
        
        // Ensure all items have a JawabanKepala record (for form consistency)
        $allItemIds = $itemsQuery->pluck('id')->toArray();
        $existingJawabanIds = $penilaian->jawabanKepala()->whereIn('instrumen_item_id', $allItemIds)->pluck('instrumen_item_id')->toArray();
        $missingItemIds = array_diff($allItemIds, $existingJawabanIds);
        
        if (!empty($missingItemIds)) {
            foreach ($missingItemIds as $itemId) {
                JawabanKepala::firstOrCreate(
                    [
                        'penilaian_kepala_id' => $penilaian->id,
                        'instrumen_item_id' => $itemId,
                    ],
                    [
                        'nilai_pengawas' => null,
                        'catatan_pengawas' => null,
                    ]
                );
            }
            // Refresh relationship after creating missing records
            $penilaian->refresh();
        }
        
        $items = $itemsQuery->paginate(10);
        $jawabanMap = $penilaian->jawabanKepala->keyBy('instrumen_item_id');
        
        // Hitung progress penilaian (hanya yang memiliki nilai valid)
        // Debug: ambil semua item ID yang seharusnya dinilai (reuse $allItemIds from above)
        
        // Debug: ambil semua jawaban yang sudah ada
        $existingJawaban = $penilaian->jawabanKepala()->get();
        $existingItemIds = $existingJawaban->pluck('instrumen_item_id')->toArray();
        
        // Debug: ambil jawaban yang sudah punya nilai valid
        $terisi = $penilaian->jawabanKepala()
            ->whereNotNull('nilai_pengawas')
            ->where('nilai_pengawas', '!=', '')
            ->where('nilai_pengawas', '>', 0)
            ->count();
        
        // Debug: item yang belum punya record
        $missingRecords = array_diff($allItemIds, $existingItemIds);
        
        // Debug: item yang punya record tapi belum ada nilai
        $missingNilai = $existingJawaban->filter(function($jawaban) {
            return empty($jawaban->nilai_pengawas) || $jawaban->nilai_pengawas <= 0;
        })->pluck('instrumen_item_id')->toArray();
        
        $isComplete = $terisi >= $totalItems;
        
        // Log untuk debug (hapus setelah fix)
        \Log::info('Penilaian Debug', [
            'penilaian_id' => $penilaian->id,
            'total_items' => $totalItems,
            'terisi' => $terisi,
            'all_item_ids' => $allItemIds,
            'existing_item_ids' => $existingItemIds,
            'missing_records' => $missingRecords,
            'missing_nilai' => $missingNilai,
            'is_complete' => $isComplete,
        ]);

        return view('pengawas.penilaian.show', compact('penilaian', 'items', 'jawabanMap', 'distribusi', 'totalItems', 'terisi', 'isComplete'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $penilaian = PenilaianKepala::where('pengawas_id', $user->id)->findOrFail($id);

        $action = $request->input('action', 'save'); // 'save' or 'finalize'

        \Log::info('Update Start', [
            'penilaian_id' => $penilaian->id,
            'action' => $action,
            'all_request_data' => $request->all(),
        ]);

        // Simpan nilai per item
        $nilai = $request->get('nilai', []);
        \Log::info('Nilai data from request', [
            'total_nilai_keys' => count($nilai),
            'nilai_keys' => array_keys($nilai),
            'nilai_data' => $nilai,
        ]);
        
        $updatedCount = 0;
        foreach ($nilai as $key => $data) {
            $itemId = $data['instrumen_item_id'] ?? null;
            // Fix: Handle empty string, "0", null, etc. as invalid values
            $rawNilai = $data['nilai_pengawas'] ?? null;
            $nilaiPengawas = null;
            if ($rawNilai !== null && $rawNilai !== '' && $rawNilai !== '0') {
                $nilaiPengawas = (int)$rawNilai;
                if ($nilaiPengawas <= 0) {
                    $nilaiPengawas = null;
                }
            }
            $catatanPengawas = isset($data['catatan_pengawas']) && $data['catatan_pengawas'] !== '' ? $data['catatan_pengawas'] : null;
            
            \Log::info('Processing nilai item', [
                'key' => $key,
                'item_id' => $itemId,
                'raw_nilai' => $rawNilai,
                'nilai_pengawas' => $nilaiPengawas,
                'catatan_pengawas' => $catatanPengawas ? substr($catatanPengawas, 0, 50) : null,
            ]);
            
            // Parse key: jika "new_" berarti belum ada record, jika angka berarti jawaban ID
            if (strpos($key, 'new_') === 0) {
                // Key format: "new_123" dimana 123 adalah item_id
                $itemId = $itemId ?? str_replace('new_', '', $key);
                
                // Create atau update berdasarkan penilaian_kepala_id dan instrumen_item_id
                if ($itemId) {
                    $existing = JawabanKepala::where('penilaian_kepala_id', $penilaian->id)
                        ->where('instrumen_item_id', $itemId)
                        ->first();
                    
                    if ($existing) {
                        // Update existing, jangan overwrite dengan null jika sudah ada nilai
                        $updateData = [];
                        if ($nilaiPengawas !== null) {
                            $updateData['nilai_pengawas'] = $nilaiPengawas;
                        }
                        if ($catatanPengawas !== null) {
                            $updateData['catatan_pengawas'] = $catatanPengawas;
                        }
                        if (!empty($updateData)) {
                            $existing->update($updateData);
                            $result = $existing->fresh();
                        } else {
                            $result = $existing;
                        }
                    } else {
                        // Create new
                        $result = JawabanKepala::create([
                            'penilaian_kepala_id' => $penilaian->id,
                            'instrumen_item_id' => $itemId,
                            'nilai_pengawas' => $nilaiPengawas,
                            'catatan_pengawas' => $catatanPengawas,
                        ]);
                    }
                    $updatedCount++;
                    \Log::info('Created/Updated with new_ key', [
                        'item_id' => $itemId,
                        'jawaban_id' => $result->id,
                        'nilai_pengawas' => $result->nilai_pengawas,
                    ]);
                }
            } else {
                // Key adalah jawaban ID (numeric)
                $jawabanId = is_numeric($key) ? $key : null;
                
                if ($jawabanId) {
                    $jawaban = JawabanKepala::where('penilaian_kepala_id', $penilaian->id)->find($jawabanId);
                    if ($jawaban) {
                        $oldNilai = $jawaban->nilai_pengawas;
                        $oldCatatan = $jawaban->catatan_pengawas;
                        
                        // Update hanya jika ada nilai baru (jangan overwrite dengan null jika sudah ada nilai)
                        $updateData = [];
                        if ($nilaiPengawas !== null) {
                            $updateData['nilai_pengawas'] = $nilaiPengawas;
                        } elseif ($oldNilai !== null) {
                            // Keep existing nilai if new value is null
                            $updateData['nilai_pengawas'] = $oldNilai;
                        }
                        
                        if ($catatanPengawas !== null) {
                            $updateData['catatan_pengawas'] = $catatanPengawas;
                        } elseif ($oldCatatan !== null) {
                            // Keep existing catatan if new value is null
                            $updateData['catatan_pengawas'] = $oldCatatan;
                        }
                        
                        if (!empty($updateData)) {
                            $jawaban->update($updateData);
                            $updatedCount++;
                            \Log::info('Updated existing jawaban', [
                                'jawaban_id' => $jawabanId,
                                'item_id' => $jawaban->instrumen_item_id,
                                'old_nilai' => $oldNilai,
                                'new_nilai' => $jawaban->fresh()->nilai_pengawas,
                                'update_data' => $updateData,
                            ]);
                        } else {
                            \Log::info('No update needed - no new data', [
                                'jawaban_id' => $jawabanId,
                                'old_nilai' => $oldNilai,
                                'new_nilai_request' => $nilaiPengawas,
                            ]);
                        }
                    } else {
                        \Log::warning('Jawaban not found', ['jawaban_id' => $jawabanId]);
                    }
                } else if ($itemId) {
                    // Fallback: jika ada item_id tapi tidak ada jawaban_id, create/update
                    $existing = JawabanKepala::where('penilaian_kepala_id', $penilaian->id)
                        ->where('instrumen_item_id', $itemId)
                        ->first();
                    
                    if ($existing) {
                        // Update existing, jangan overwrite dengan null jika sudah ada nilai
                        $updateData = [];
                        if ($nilaiPengawas !== null) {
                            $updateData['nilai_pengawas'] = $nilaiPengawas;
                        }
                        if ($catatanPengawas !== null) {
                            $updateData['catatan_pengawas'] = $catatanPengawas;
                        }
                        if (!empty($updateData)) {
                            $existing->update($updateData);
                            $result = $existing->fresh();
                        } else {
                            $result = $existing;
                        }
                    } else {
                        // Create new
                        $result = JawabanKepala::create([
                            'penilaian_kepala_id' => $penilaian->id,
                            'instrumen_item_id' => $itemId,
                            'nilai_pengawas' => $nilaiPengawas,
                            'catatan_pengawas' => $catatanPengawas,
                        ]);
                    }
                    $updatedCount++;
                    \Log::info('Created/Updated with fallback', [
                        'item_id' => $itemId,
                        'jawaban_id' => $result->id,
                        'nilai_pengawas' => $result->nilai_pengawas,
                    ]);
                } else {
                    \Log::warning('Cannot process - no jawaban_id or item_id', ['key' => $key, 'data' => $data]);
                }
            }
        }
        
        \Log::info('Update completed', [
            'penilaian_id' => $penilaian->id,
            'updated_count' => $updatedCount,
            'total_nilai_keys' => count($nilai),
        ]);

        // Refresh penilaian untuk mendapatkan data terbaru setelah update
        $penilaian->refresh();
        
        // Pastikan semua item punya record JawabanKepala (untuk pagination)
        // Jika ada item yang belum punya record, buat record kosong
        $distribusi = $penilaian->distribusiInstrumen;
        if ($distribusi->komponen) {
            $allItemIds = $distribusi->komponen->instrumenPenilaians()->pluck('id')->toArray();
        } else {
            $subIds = $distribusi->instrumen->komponenPenilaians->flatMap(function ($k) {
                return $k->subKomponens->pluck('id');
            })->values();
            $allItemIds = InstrumenPenilaian::whereIn('sub_komponen_id', $subIds)->pluck('id')->toArray();
        }
        
        $existingItemIds = $penilaian->jawabanKepala()->pluck('instrumen_item_id')->toArray();
        $missingItemIds = array_diff($allItemIds, $existingItemIds);
        
        \Log::info('Update Debug - Before create missing records', [
            'penilaian_id' => $penilaian->id,
            'total_items' => count($allItemIds),
            'existing_items' => count($existingItemIds),
            'missing_items' => count($missingItemIds),
            'missing_item_ids' => array_values($missingItemIds),
        ]);
        
        // Buat record kosong untuk item yang belum punya record
        foreach ($missingItemIds as $itemId) {
            JawabanKepala::firstOrCreate([
                'penilaian_kepala_id' => $penilaian->id,
                'instrumen_item_id' => $itemId,
            ]);
        }
        
        // Refresh lagi setelah membuat record baru
        $penilaian->refresh();
        
        // Cek setelah refresh
        $allJawaban = $penilaian->jawabanKepala()->get();
        $terisiCount = $allJawaban->filter(function($j) {
            return !empty($j->nilai_pengawas) && $j->nilai_pengawas > 0;
        })->count();
        
        $missingNilaiItems = $allJawaban->filter(function($j) {
            return empty($j->nilai_pengawas) || $j->nilai_pengawas <= 0;
        })->pluck('instrumen_item_id')->toArray();
        
        \Log::info('Update Debug - After refresh', [
            'penilaian_id' => $penilaian->id,
            'total_items' => count($allItemIds),
            'terisi_count' => $terisiCount,
            'missing_nilai_items' => $missingNilaiItems,
            'all_jawaban_count' => $allJawaban->count(),
        ]);

        // Jika finalisasi, validasi semua item sudah dinilai
        if ($action === 'finalize') {
            $distribusi = $penilaian->distribusiInstrumen;
            if ($distribusi->komponen) {
                $totalItems = $distribusi->komponen->instrumenPenilaians()->count();
            } else {
                $subIds = $distribusi->instrumen->komponenPenilaians->flatMap(function ($k) {
                    return $k->subKomponens->pluck('id');
                })->values();
                $totalItems = InstrumenPenilaian::whereIn('sub_komponen_id', $subIds)->count();
            }
            
            // Refresh untuk mendapatkan data terbaru
            $penilaian->refresh();
            
            // Ambil semua item ID yang seharusnya dinilai
            $allItemIds = $distribusi->komponen 
                ? $distribusi->komponen->instrumenPenilaians()->pluck('id')->toArray()
                : InstrumenPenilaian::whereIn('sub_komponen_id', $subIds)->pluck('id')->toArray();
            
            // Validasi: semua item harus punya nilai_pengawas yang tidak null dan tidak kosong
            $terisi = $penilaian->jawabanKepala()
                ->whereNotNull('nilai_pengawas')
                ->where('nilai_pengawas', '!=', '')
                ->where('nilai_pengawas', '>', 0)
                ->count();
            
            // Debug: cek item yang belum dinilai
            $allJawaban = $penilaian->jawabanKepala()->get();
            $existingItemIds = $allJawaban->pluck('instrumen_item_id')->toArray();
            $missingRecords = array_diff($allItemIds, $existingItemIds);
            $missingNilai = $allJawaban->filter(function($jawaban) {
                return empty($jawaban->nilai_pengawas) || $jawaban->nilai_pengawas <= 0;
            })->pluck('instrumen_item_id')->toArray();
            
            \Log::info('Finalisasi Debug', [
                'penilaian_id' => $penilaian->id,
                'total_items' => $totalItems,
                'terisi' => $terisi,
                'all_item_ids' => $allItemIds,
                'existing_item_ids' => $existingItemIds,
                'missing_records' => $missingRecords,
                'missing_nilai' => $missingNilai,
            ]);
            
            if ($terisi < $totalItems) {
                $missingCount = count($missingRecords) + count($missingNilai);
                return back()->with('error', "Anda harus menilai semua item terlebih dahulu ({$terisi}/{$totalItems} sudah dinilai). Missing records: " . count($missingRecords) . ", Missing nilai: " . count($missingNilai) . ". Pastikan semua item memiliki nilai pengawas yang valid (1-5).");
            }

            // Validasi status finalisasi harus dipilih
            $finalStatus = $request->input('final_status');
            if (empty($finalStatus) || !in_array($finalStatus, ['validated', 'final'])) {
                return back()->with('error', 'Silakan pilih Status Finalisasi (Validated atau Final) sebelum melakukan finalisasi.');
            }

            // AUTO-GENERATE AI untuk Final dan Validated (WAJIB saat finalisasi)
            // Analisis AI lengkap di-generate saat finalisasi agar:
            // 1. Data sudah final dan lengkap
            // 2. PDF generation jadi lebih cepat (hanya load dari DB)
            // 3. Konsisten - sekali generate, bisa digunakan berkali-kali
            $catatanManual = $request->get('catatan_pengawas') ?: '';
            $catatanAI = null;
            $aiGenerated = false;
            
            try {
                $aiService = new AIAnalysisService();
                if ($aiService->isEnabled()) {
                    // Generate AI analysis (ini bisa memakan waktu 10-30 detik untuk analisis lengkap)
                    \Log::info('Generating AI analysis for finalization', [
                        'penilaian_id' => $penilaian->id,
                        'status' => $finalStatus
                    ]);
                    
                    $catatanAI = $aiService->generatePengawasAnalysisKepala($penilaian);
                    $aiGenerated = true;
                    
                    \Log::info('AI analysis generated successfully', [
                        'penilaian_id' => $penilaian->id,
                        'ai_length' => strlen($catatanAI)
                    ]);
                    
                    // Gunakan AI sebagai catatan utama, append catatan manual jika ada
                    if (!empty(trim($catatanManual))) {
                        $catatan = $catatanManual . "\n\n---\n\nANALISIS AI PENGAWAS:\n" . $catatanAI;
                    } else {
                        $catatan = $catatanAI;
                    }
                } else {
                    // AI tidak diaktifkan, gunakan catatan manual atau warning
                    if (!empty(trim($catatanManual))) {
                        $catatan = $catatanManual;
                    } else {
                        $catatan = 'Analisis AI tidak diaktifkan. Silakan hubungi administrator untuk mengaktifkan fitur AI atau tambahkan catatan manual.';
                    }
                }
            } catch (\Throwable $e) {
                \Log::error('AI Analysis generation failed during finalization', [
                    'penilaian_id' => $penilaian->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                
                // Jika AI gagal, gunakan catatan manual atau beri warning
                if (!empty(trim($catatanManual))) {
                    $catatan = $catatanManual . "\n\n[Catatan: Analisis AI gagal di-generate: " . $e->getMessage() . "]";
                } else {
                    return back()->with('error', 'Gagal menghasilkan analisis AI. Silakan coba lagi atau tambahkan catatan manual terlebih dahulu. Error: ' . $e->getMessage());
                }
            }

            $updateData = [
                'status' => $finalStatus,
                'tanggal_validasi' => now(),
            ];
            
            // Simpan catatan (AI atau manual) ke database
            if (!empty(trim($catatan))) {
                $updateData['catatan_pengawas'] = $catatan;
            }
            
            $penilaian->update($updateData);

            $successMessage = 'Penilaian berhasil difinalisasi dengan status: ' . ucfirst($finalStatus);
            if ($aiGenerated) {
                $successMessage .= '. Analisis AI telah di-generate dan tersimpan.';
            }

            return back()->with('success', $successMessage);
        } else {
            // Simpan draft - manual AI hanya jika checkbox dicentang (opsional)
            $catatan = $request->get('catatan_pengawas') ?: null;
            if ($request->get('generate_ai') == '1') {
                try {
                    $ai = new AIAnalysisService();
                    if ($ai->isEnabled()) {
                        $catatan = $ai->generatePengawasAnalysisKepala($penilaian);
                    }
                } catch (\Throwable $e) {
                    // keep manual catatan atau null
                }
            }

            // Update status jika diubah manual (untuk revision, dll)
            $update = [];
            if (!empty(trim($catatan))) {
                $update['catatan_pengawas'] = $catatan;
            } elseif ($catatan === '') {
                $update['catatan_pengawas'] = null;
            }
            if ($request->filled('status') && in_array($request->status, ['draft', 'revision'])) {
                $update['status'] = $request->status;
            }
            
            if (!empty($update)) {
                $penilaian->update($update);
            }

            return back()->with('success', 'Penilaian draft berhasil disimpan.');
        }
    }

    public function submit($id)
    {
        $user = Auth::user();
        $penilaian = PenilaianKepala::where('pengawas_id', $user->id)->findOrFail($id);
        $penilaian->update([
            'status' => 'submitted',
            'tanggal_submit' => now(),
        ]);
        return redirect()->route('pengawas.penilaian.index')->with('success', 'Penilaian dikirim');
    }

    /**
     * Debug method untuk melihat detail penilaian
     */
    public function debug($id)
    {
        $user = Auth::user();
        $penilaian = PenilaianKepala::with(['jawabanKepala', 'kepala', 'distribusiInstrumen'])
            ->where('pengawas_id', $user->id)
            ->findOrFail($id);

        $distribusi = $penilaian->distribusiInstrumen;
        if ($distribusi->komponen) {
            // Fix: Add secondary sort by id to ensure consistent ordering when no_urutan is duplicate
            $itemsQuery = $distribusi->komponen->instrumenPenilaians()->orderBy('no_urutan')->orderBy('id');
        } else {
            $subIds = $distribusi->instrumen->komponenPenilaians->flatMap(function ($k) {
                return $k->subKomponens->pluck('id');
            })->values();
            // Fix: Add secondary sort by id to ensure consistent ordering when no_urutan is duplicate
            $itemsQuery = InstrumenPenilaian::whereIn('sub_komponen_id', $subIds)->orderBy('no_urutan')->orderBy('id');
        }
        
        $totalItems = $itemsQuery->count();
        $allItemIds = $itemsQuery->pluck('id')->toArray();
        
        $allJawaban = $penilaian->jawabanKepala()->get();
        $existingItemIds = $allJawaban->pluck('instrumen_item_id')->toArray();
        $missingRecords = array_diff($allItemIds, $existingItemIds);
        
        $terisi = $penilaian->jawabanKepala()
            ->whereNotNull('nilai_pengawas')
            ->where('nilai_pengawas', '!=', '')
            ->where('nilai_pengawas', '>', 0)
            ->count();
        
        $missingNilai = $allJawaban->filter(function($jawaban) {
            return empty($jawaban->nilai_pengawas) || $jawaban->nilai_pengawas <= 0;
        })->map(function($jawaban) {
            return [
                'jawaban_id' => $jawaban->id,
                'item_id' => $jawaban->instrumen_item_id,
                'nilai_pengawas' => $jawaban->nilai_pengawas,
                'catatan_pengawas' => $jawaban->catatan_pengawas,
            ];
        });
        
        // Get item details untuk missing nilai dan hitung halaman pagination
        $perPage = 10; // Same as pagination in show method
        $missingNilaiDetails = [];
        foreach ($missingNilai as $missing) {
            $item = InstrumenPenilaian::find($missing['item_id']);
            if ($item) {
                // Hitung posisi item dalam urutan untuk menentukan halaman
                $itemPosition = array_search($missing['item_id'], $allItemIds);
                $pageNumber = $itemPosition !== false ? (int)ceil(($itemPosition + 1) / $perPage) : null;
                
                $missingNilaiDetails[] = [
                    'jawaban_id' => $missing['jawaban_id'],
                    'item_id' => $missing['item_id'],
                    'no_urutan' => $item->no_urutan ?? null,
                    'indikator_kinerja' => $item->indikator_kinerja ?? null,
                    'nilai_pengawas' => $missing['nilai_pengawas'],
                    'catatan_pengawas' => $missing['catatan_pengawas'],
                    'position' => $itemPosition !== false ? $itemPosition + 1 : null,
                    'page_number' => $pageNumber,
                ];
            } else {
                $missingNilaiDetails[] = $missing;
            }
        }
        
        return response()->json([
            'penilaian_id' => $penilaian->id,
            'total_items' => $totalItems,
            'terisi' => $terisi,
            'all_item_ids' => $allItemIds,
            'existing_item_ids' => $existingItemIds,
            'missing_records' => array_values($missingRecords),
            'missing_nilai' => $missingNilaiDetails,
            'all_jawaban' => $allJawaban->map(function($j) {
                return [
                    'id' => $j->id,
                    'item_id' => $j->instrumen_item_id,
                    'nilai_pengawas' => $j->nilai_pengawas,
                    'has_value' => !empty($j->nilai_pengawas) && $j->nilai_pengawas > 0,
                ];
            }),
        ]);
    }

    /**
     * Ubah status penilaian kembali ke draft agar bisa diedit ulang
     */
    public function ubahStatus($id, Request $request)
    {
        $user = Auth::user();
        
        $penilaian = PenilaianKepala::where('pengawas_id', $user->id)
            ->findOrFail($id);

        // Hanya bisa ubah status jika sudah validated atau final
        if (!in_array($penilaian->status, ['validated', 'final'])) {
            return back()->with('error', 'Status hanya bisa diubah jika penilaian sudah divalidasi atau final.');
        }

        $newStatus = $request->input('status', 'draft');
        
        // Validasi status yang diizinkan
        $allowedStatuses = ['draft', 'revision'];
        if (!in_array($newStatus, $allowedStatuses)) {
            return back()->with('error', 'Status tidak valid. Hanya boleh diubah ke draft atau revision.');
        }

        // Ubah status dan reset tanggal validasi
        $updateData = [
            'status' => $newStatus,
            'tanggal_validasi' => null,
        ];

        // Optional: clear catatan_pengawas jika ingin reset analisis AI juga
        // Atau biarkan catatan tetap ada sebagai referensi
        // $updateData['catatan_pengawas'] = null;

        $penilaian->update($updateData);

        $statusText = $newStatus == 'draft' ? 'draft' : 'perlu direvisi';
        return back()->with('success', 'Status penilaian berhasil diubah menjadi ' . $statusText . '. Anda sekarang dapat mengedit penilaian kembali.');
    }

    /**
     * Kirim raport ke kepala (notifikasi bahwa raport sudah tersedia)
     */
    public function kirimRaportKepala($id, Request $request)
    {
        $user = Auth::user();
        
        $penilaian = PenilaianKepala::where('pengawas_id', $user->id)
            ->findOrFail($id);

        // Hanya bisa kirim jika status validated atau final
        if (!in_array($penilaian->status, ['validated', 'final'])) {
            return back()->with('error', 'Raport hanya dapat dikirim untuk penilaian berstatus validated atau final.');
        }

        // Buat notifikasi untuk kepala
        try {
            DB::table('notifications')->insert([
                'type' => 'raport_penilaian_kepala',
                'title' => 'Raport Penilaian Tersedia',
                'message' => 'Raport penilaian Anda sudah tersedia. Silakan unduh melalui halaman penilaian.',
                'user_id' => $penilaian->kepala_id,
                'related_id' => $penilaian->id,
                'related_type' => 'App\Models\PenilaianKepala',
                'is_read' => false,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            return back()->with('success', 'Raport berhasil dikirim ke kepala sekolah. Kepala akan menerima notifikasi.');
        } catch (\Exception $e) {
            \Log::error('Error sending raport notification: ' . $e->getMessage());
            return back()->with('error', 'Gagal mengirim notifikasi. Error: ' . $e->getMessage());
        }
    }
}


