<?php

namespace App\Http\Controllers\Pengawas;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Lembaga;

class LembagaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        $query = Lembaga::where('pengawas_id', $user->id);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('nama_lembaga', 'like', "%{$search}%")
                  ->orWhere('npsn_nsm', 'like', "%{$search}%")
                  ->orWhere('nama_kepala', 'like', "%{$search}%");
            });
        }

        if ($request->filled('jenis')) {
            $query->where('jenis', $request->jenis);
        }

        $lembaga = $query->with('kepala')->orderBy('nama_lembaga')->paginate(15);

        return view('pengawas.lembaga.index', compact('lembaga'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Get lembaga yang belum memiliki pengawas
        $lembagaAvailable = Lembaga::whereNull('pengawas_id')
            ->orderBy('nama_lembaga')
            ->get();
            
        return view('pengawas.lembaga.create', compact('lembagaAvailable'));
    }

    /**
     * Claim a lembaga (assign pengawas to existing lembaga).
     */
    public function claim(Request $request)
    {
        $validated = $request->validate([
            'lembaga_id' => 'required|exists:lembaga,id'
        ]);

        $lembaga = Lembaga::findOrFail($validated['lembaga_id']);
        
        // Check if lembaga is already claimed by another pengawas
        if ($lembaga->pengawas_id !== null) {
            return redirect()->back()
                ->with('error', 'Lembaga ini sudah dibina oleh pengawas lain.');
        }
        
        // Check if current pengawas already has this lembaga
        $existingClaim = Lembaga::where('pengawas_id', auth()->id())
            ->where('npsn_nsm', $lembaga->npsn_nsm)
            ->first();
            
        if ($existingClaim) {
            return redirect()->back()
                ->with('error', 'Anda sudah memiliki lembaga dengan NSM yang sama.');
        }

        // Claim the lembaga
        $lembaga->update(['pengawas_id' => auth()->id()]);

        return redirect()->route('pengawas.lembaga.index')
            ->with('success', 'Lembaga "' . $lembaga->nama_lembaga . '" berhasil diambil sebagai lembaga binaan!');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $userId = Auth::id();
        $lembaga = Lembaga::where('pengawas_id', $userId)
            ->with(['guru.user'])
            ->findOrFail($id);

        return view('pengawas.lembaga.show', compact('lembaga'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $userId = Auth::id();
        $lembaga = Lembaga::where('pengawas_id', $userId)->findOrFail($id);

        return view('pengawas.lembaga.edit', compact('lembaga'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $userId = Auth::id();
        $lembaga = Lembaga::where('pengawas_id', $userId)->findOrFail($id);

        $validated = $request->validate([
            'npsn_nsm' => [
                'required',
                'string',
                'max:20',
                Rule::unique('lembaga', 'npsn_nsm')
                    ->where(function ($query) {
                        return $query->where('pengawas_id', auth()->id());
                    })
                    ->ignore($lembaga->id)
            ],
            'nama_lembaga' => 'required|string|max:255',
            'jenis' => 'required|in:RA,MI,MTs,MA',
            'alamat_lengkap' => 'required|string',
            'kelurahan' => 'required|string|max:100',
            'kecamatan' => 'required|string|max:100',
            'kabupaten' => 'required|string|max:100',
            'provinsi' => 'required|string|max:100',
            'kode_pos' => 'nullable|string|max:10',
            'telepon' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|url|max:255',
            'nama_kepala' => 'required|string|max:255',
            'nip_kepala' => 'required|string|max:50',
            'akreditasi' => 'nullable|string|max:10',
            'tahun_berdiri' => 'nullable|integer|min:1900|max:' . date('Y'),
        ]);

        $lembaga->update($validated);

        return redirect()->route('pengawas.lembaga.index')
            ->with('status', 'Lembaga berhasil diperbarui!');
    }

    /**
     * Remove lembaga from pengawas binaan (set pengawas_id to null).
     */
    public function destroy(string $id)
    {
        $userId = Auth::id();
        $lembaga = Lembaga::where('pengawas_id', $userId)->findOrFail($id);

        // Check if lembaga has associated teachers
        if ($lembaga->guru()->count() > 0) {
            return redirect()->route('pengawas.lembaga.index')
                ->with('error', 'Tidak dapat melepaskan lembaga yang memiliki guru. Hapus semua guru terlebih dahulu.');
        }

        // Set pengawas_id to null instead of deleting the lembaga
        $lembaga->update(['pengawas_id' => null]);

        return redirect()->route('pengawas.lembaga.index')
            ->with('status', 'Lembaga berhasil dilepaskan dari binaan!');
    }
}