<?php

namespace App\Http\Controllers\Pengawas;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\DistribusiInstrumen;
use App\Models\Instrumen;
use App\Models\KomponenPenilaian;
use App\Models\Lembaga;
use App\Models\PenilaianKepala;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DistribusiController extends Controller
{
    /**
     * Display a listing of distributed instruments
     */
    public function index()
    {
        $user = Auth::user();
        
        // Get all distributed instruments for pengawas's schools
        $distribusi = DistribusiInstrumen::with(['instrumen', 'komponen', 'lembaga', 'penilaianKepala'])
            ->whereHas('lembaga', function($query) use ($user) {
                $query->where('pengawas_id', $user->id);
            })
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        // Get statistics
        $stats = [
            'total_distribusi' => $distribusi->total(),
            'active_distribusi' => DistribusiInstrumen::whereHas('lembaga', function($query) use ($user) {
                $query->where('pengawas_id', $user->id);
            })->where('status', 'active')->count(),
            'completed_penilaian' => DistribusiInstrumen::whereHas('lembaga', function($query) use ($user) {
                $query->where('pengawas_id', $user->id);
            })->whereHas('penilaianKepala', function($query) use ($user) {
                $query->where('pengawas_id', $user->id)->whereIn('status', ['validated', 'final']);
            })->count(),
            'in_progress' => PenilaianKepala::whereHas('distribusiInstrumen.lembaga', function($query) use ($user) {
                $query->where('pengawas_id', $user->id);
            })->where('pengawas_id', $user->id)->whereIn('status', ['draft', 'submitted', 'revision'])->count(),
        ];

        return view('pengawas.distribusi.index', compact('distribusi', 'stats'));
    }

    /**
     * Show the form for creating a new distribution
     */
    public function create()
    {
        $user = Auth::user();
        
        // Get available instruments
        $instrumen = Instrumen::where('is_active', true)->get();
        
        // Get pengawas's schools
        $lembaga = Lembaga::where('pengawas_id', $user->id)->get();

        return view('pengawas.distribusi.create', compact('instrumen', 'lembaga'));
    }

    /**
     * Store a newly created distribution
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        
        $validated = $request->validate([
            'instrumen_id' => 'required|exists:instrumen,id',
            'jenis_distribusi' => 'required|in:semua_komponen,per_komponen',
            'komponen_ids' => 'required_if:jenis_distribusi,per_komponen|array',
            'komponen_ids.*' => 'exists:komponen_penilaians,id',
            'lembaga_ids' => 'required|array',
            'lembaga_ids.*' => 'exists:lembaga,id',
            'deadline' => 'required|date|after:today',
        ]);

        DB::beginTransaction();
        try {
            // Determine which komponen to distribute
            if ($validated['jenis_distribusi'] === 'semua_komponen') {
                // Get all komponen for this instrumen
                $komponenIds = KomponenPenilaian::where('instrumen_id', $validated['instrumen_id'])
                    ->pluck('id')
                    ->toArray();
            } else {
                // Use selected komponen
                $komponenIds = $validated['komponen_ids'];
            }

            foreach ($validated['lembaga_ids'] as $lembagaId) {
                // Check if lembaga belongs to this pengawas
                $lembaga = Lembaga::where('id', $lembagaId)
                    ->where('pengawas_id', $user->id)
                    ->first();
                
                if (!$lembaga) {
                    continue; // Skip if lembaga doesn't belong to pengawas
                }

                // Create distribution for each komponen
                foreach ($komponenIds as $komponenId) {
                    $distribusi = DistribusiInstrumen::create([
                        'instrumen_id' => $validated['instrumen_id'],
                        'komponen_id' => $komponenId,
                        'lembaga_id' => $lembagaId,
                        'tanggal_distribusi' => now()->format('Y-m-d'),
                        'deadline' => $validated['deadline'],
                        'status' => 'active',
                        'jenis_distribusi' => $validated['jenis_distribusi'],
                    ]);

                // Create PenilaianKepala for kepala lembaga ini
                PenilaianKepala::create([
                    'distribusi_instrumen_id' => $distribusi->id,
                    'kepala_id' => $lembaga->kepala_id,
                    'pengawas_id' => $user->id,
                    'status' => 'draft',
                ]);
                }
            }

            DB::commit();
            return redirect()->route('pengawas.distribusi.index')
                ->with('success', 'Instrumen berhasil didistribusikan ke ' . count($validated['lembaga_ids']) . ' lembaga.');
                
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()
                ->with('error', 'Terjadi kesalahan saat mendistribusikan instrumen: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified distribution
     */
    public function show($id)
    {
        $user = Auth::user();
        
        $distribusi = DistribusiInstrumen::with(['instrumen', 'komponen', 'lembaga', 'penilaianKepala.kepala', 'penilaianKepala.pengawas'])
            ->whereHas('lembaga', function($query) use ($user) {
                $query->where('pengawas_id', $user->id);
            })
            ->findOrFail($id);

        // Get penilaian statistics
        $penilaianStats = [
            'total_kepala' => $distribusi->penilaianKepala->count(),
            'total' => $distribusi->penilaianKepala->count(),
            'draft' => $distribusi->penilaianKepala->where('status', 'draft')->count(),
            'submitted' => $distribusi->penilaianKepala->where('status', 'submitted')->count(),
            'revision' => $distribusi->penilaianKepala->where('status', 'revision')->count(),
            'validated' => $distribusi->penilaianKepala->where('status', 'validated')->count() + $distribusi->penilaianKepala->where('status', 'final')->count(),
            'final' => $distribusi->penilaianKepala->where('status', 'final')->count(),
        ];

        return view('pengawas.distribusi.show', compact('distribusi', 'penilaianStats'));
    }

    /**
     * Get komponen based on selected instrumen (AJAX)
     */
    public function getKomponen(Request $request)
    {
        // Manual authentication check
        if (!Auth::check() || !Auth::user()->hasRole('pengawas')) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }
        
        $instrumenId = $request->input('instrumen_id');
        
        if (!$instrumenId) {
            return response()->json([]);
        }

        $komponen = KomponenPenilaian::where('instrumen_id', $instrumenId)
            ->orderBy('urutan')
            ->get();

        return response()->json($komponen);
    }

    /**
     * Close/Complete a distribution
     */
    public function close($id)
    {
        $user = Auth::user();
        
        $distribusi = DistribusiInstrumen::whereHas('lembaga', function($query) use ($user) {
            $query->where('pengawas_id', $user->id);
        })->findOrFail($id);

        $distribusi->update(['status' => 'closed']);

        return redirect()->route('pengawas.distribusi.show', $id)
            ->with('success', 'Distribusi instrumen telah ditutup.');
    }
}
