<?php

namespace App\Http\Controllers\Kepala;

use App\Http\Controllers\Controller;
use App\Models\PenilaianGuru;
use App\Models\PenilaianKepala;
use App\Services\AIAnalysisService;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;

class RaportController extends Controller
{
    /**
     * Generate PDF raport for penilaian guru
     */
    public function generateRaport($id)
    {
        try {
            $user = Auth::user();
            
            // Get penilaian data with all relationships
            $penilaianGuru = PenilaianGuru::with([
                'guru.lembaga',
                'kepala',
                'distribusiInstrumen.instrumen',
                'distribusiInstrumen.komponen',
                'jawabanGuru.instrumenItem.subKomponen.komponenPenilaian',
                'validasiPengawas.pengawas'
            ])
            ->where('kepala_id', $user->id)
            ->findOrFail($id);

        // Check if penilaian is validated or final
        if (!in_array($penilaianGuru->status, ['validated', 'final'])) {
            return redirect()->back()
                ->with('error', 'Raport hanya dapat dihasilkan untuk penilaian yang sudah divalidasi.');
        }

        // Get jawaban guru with instrumen items sorted by sub komponen
        $jawabanGuru = $penilaianGuru->jawabanGuru()
            ->leftJoin('instrumen_penilaians', 'jawaban_guru.instrumen_item_id', '=', 'instrumen_penilaians.id')
            ->leftJoin('sub_komponens', 'instrumen_penilaians.sub_komponen_id', '=', 'sub_komponens.id')
            ->leftJoin('komponen_penilaians', 'sub_komponens.komponen_penilaian_id', '=', 'komponen_penilaians.id')
            ->with('instrumenItem')
            ->orderBy('komponen_penilaians.urutan', 'asc')
            ->orderBy('sub_komponens.urutan', 'asc')
            ->orderBy('instrumen_penilaians.no_urutan', 'asc')
            ->select('jawaban_guru.*')
            ->get();

        // Group jawaban by sub komponen for analysis
        $subKomponenAnalysis = $this->prepareSubKomponenAnalysis($jawabanGuru);

        // Generate pengawas analysis if available
        $analisisPengawas = null;
        if ($penilaianGuru->validasiPengawas) {
            try {
                $aiService = new AIAnalysisService();
                $analisisPengawas = $aiService->generatePengawasAnalysis($penilaianGuru);
            } catch (\Exception $e) {
                \Log::error('Pengawas AI Analysis Error: ' . $e->getMessage());
            }
        }

        // Regenerate kepala analysis if not available or is placeholder
        $analisisKepala = $penilaianGuru->catatan_kepala;
        if (!$analisisKepala || $analisisKepala === 'Belum ada analisis kepala sekolah.' || strpos($analisisKepala, 'Terjadi kesalahan') !== false) {
            try {
                $aiService = new AIAnalysisService();
                
                // Check if AI is enabled
                if (!$aiService->isEnabled()) {
                    $analisisKepala = 'Analisis AI tidak diaktifkan. Silakan hubungi administrator untuk mengaktifkan fitur AI.';
                } else {
                    $analisisKepala = $aiService->generateKepalaAnalysis($penilaianGuru);
                    
                    // Update the database with new analysis
                    $penilaianGuru->update(['catatan_kepala' => $analisisKepala]);
                }
            } catch (\Exception $e) {
                \Log::error('Kepala AI Analysis Error: ' . $e->getMessage());
                \Log::error('AI Analysis Stack Trace: ' . $e->getTraceAsString());
                $analisisKepala = 'Terjadi kesalahan dalam menghasilkan analisis. Silakan coba lagi.';
            }
        }

        // Prepare data for PDF with actual database values
        $lembaga = $penilaianGuru->guru?->lembaga;
        $guru = $penilaianGuru->guru;
        $kepala = $penilaianGuru->kepala;
        $instrumen = $penilaianGuru->distribusiInstrumen?->instrumen;
        $komponen = $penilaianGuru->distribusiInstrumen?->komponen;
        
        $data = [
            'penilaianGuru' => $penilaianGuru,
            'jawabanGuru' => $jawabanGuru,
            'subKomponenAnalysis' => $subKomponenAnalysis,
            'analisisKepala' => $analisisKepala,
            'analisisPengawas' => $analisisPengawas,
            'lembaga' => $lembaga,
            'kepala' => $kepala,
            'guru' => $guru,
            'instrumen' => $instrumen,
            'komponen' => $komponen,
            'pengawas' => $penilaianGuru->validasiPengawas?->pengawas,
            'tanggal' => now()->format('d F Y'),
            'tahun' => now()->year,
            'kabupaten' => $this->formatKabupaten($lembaga?->kabupaten_name ?? $lembaga?->kabupaten),
        ];

        // Log data for debugging
        \Log::info('PDF Generation Data:', [
            'penilaian_id' => $penilaianGuru->id,
            'status' => $penilaianGuru->status,
            'guru_name' => $penilaianGuru->guru?->nama,
            'guru_full_name' => $penilaianGuru->guru?->nama_lengkap,
            'lembaga_name' => $penilaianGuru->guru?->lembaga?->nama_lembaga,
            'kepala_name' => $penilaianGuru->kepala?->name,
            'jawaban_count' => $jawabanGuru->count(),
            'sub_komponen_count' => count($subKomponenAnalysis),
            'analisis_kepala_length' => strlen($analisisKepala ?? ''),
            'kabupaten' => $lembaga?->kabupaten_name ?? $lembaga?->kabupaten
        ]);

        // Generate PDF
        $pdf = Pdf::loadView('kepala.raport.template', $data);
        $pdf->setPaper('A4', 'portrait');
        
        $filename = 'Raport_Penilaian_' . $penilaianGuru->guru->nama . '_' . now()->format('Y-m-d') . '.pdf';
        
        return $pdf->download($filename);
        
        } catch (\Exception $e) {
            \Log::error('PDF Generation Error: ' . $e->getMessage());
            \Log::error('PDF Generation Stack Trace: ' . $e->getTraceAsString());
            
            return redirect()->back()
                ->with('error', 'Terjadi kesalahan saat menghasilkan PDF. Error: ' . $e->getMessage());
        }
    }

    /**
     * Prepare sub komponen analysis data
     */
    private function prepareSubKomponenAnalysis($jawabanGuru)
    {
        $groupedBySubKomponen = $jawabanGuru->groupBy('instrumenItem.sub_komponen_id');
        $analysis = [];

        foreach ($groupedBySubKomponen as $subKomponenId => $jawabanGroup) {
            $subKomponen = $jawabanGroup->first()->instrumenItem->subKomponen;
            $nilaiKepala = $jawabanGroup->pluck('nilai_kepala')->filter()->values();
            
            if ($nilaiKepala->isNotEmpty()) {
                $rataRata = $nilaiKepala->avg();
                $kategori = $this->determineCategory($rataRata);
                
                $analysis[] = [
                    'nama_sub_komponen' => $subKomponen->nama_sub_komponen,
                    'deskripsi' => $subKomponen->deskripsi,
                    'rata_rata' => round($rataRata, 2),
                    'kategori' => $kategori,
                    'nilai_detail' => $nilaiKepala->toArray(),
                    'indikator_kinerja' => $jawabanGroup->map(function($jawaban) {
                        return [
                            'indikator' => $jawaban->instrumenItem->indikator_kinerja,
                            'nilai' => $jawaban->nilai_kepala,
                            'bukti' => $jawaban->keterangan_bukti
                        ];
                    })->toArray()
                ];
            }
        }

        return $analysis;
    }

    /**
     * Debug raport data (for testing)
     */
    public function debugRaport($id)
    {
        try {
            $user = Auth::user();
            
            // Get penilaian data with all relationships
            $penilaianGuru = PenilaianGuru::with([
                'guru.lembaga',
                'kepala',
                'distribusiInstrumen.instrumen',
                'distribusiInstrumen.komponen',
                'jawabanGuru.instrumenItem.subKomponen.komponenPenilaian',
                'validasiPengawas.pengawas'
            ])
            ->where('kepala_id', $user->id)
            ->findOrFail($id);

            // Check if penilaian is validated or final
            if (!in_array($penilaianGuru->status, ['validated', 'final'])) {
                return response()->json([
                    'error' => 'Raport hanya dapat dihasilkan untuk penilaian yang sudah divalidasi.',
                    'status' => $penilaianGuru->status
                ]);
            }

            // Get jawaban guru
            $jawabanGuru = $penilaianGuru->jawabanGuru()
                ->leftJoin('instrumen_penilaians', 'jawaban_guru.instrumen_item_id', '=', 'instrumen_penilaians.id')
                ->leftJoin('sub_komponens', 'instrumen_penilaians.sub_komponen_id', '=', 'sub_komponens.id')
                ->leftJoin('komponen_penilaians', 'sub_komponens.komponen_penilaian_id', '=', 'komponen_penilaians.id')
                ->with('instrumenItem')
                ->orderBy('komponen_penilaians.urutan', 'asc')
                ->orderBy('sub_komponens.urutan', 'asc')
                ->orderBy('instrumen_penilaians.no_urutan', 'asc')
                ->select('jawaban_guru.*')
                ->get();

            // Prepare sub komponen analysis
            $subKomponenAnalysis = $this->prepareSubKomponenAnalysis($jawabanGuru);

            return response()->json([
                'success' => true,
                'data' => [
                    'penilaian_id' => $penilaianGuru->id,
                    'status' => $penilaianGuru->status,
                    'guru' => [
                        'id' => $penilaianGuru->guru?->id,
                        'nama' => $penilaianGuru->guru?->nama,
                        'mata_pelajaran' => $penilaianGuru->guru?->mata_pelajaran,
                    ],
                    'lembaga' => [
                        'id' => $penilaianGuru->guru?->lembaga?->id,
                        'nama_lembaga' => $penilaianGuru?->lembaga?->nama_lembaga,
                    ],
                    'kepala' => [
                        'id' => $penilaianGuru->kepala?->id,
                        'name' => $penilaianGuru->kepala?->name,
                    ],
                    'instrumen' => [
                        'id' => $penilaianGuru->distribusiInstrumen?->instrumen?->id,
                        'nama_instrumen' => $penilaianGuru->distribusiInstrumen?->instrumen?->nama_instrumen,
                    ],
                    'komponen' => [
                        'id' => $penilaianGuru->distribusiInstrumen?->komponen?->id,
                        'nama_komponen' => $penilaianGuru->distribusiInstrumen?->komponen?->nama_komponen,
                    ],
                    'jawaban_count' => $jawabanGuru->count(),
                    'sub_komponen_analysis_count' => count($subKomponenAnalysis),
                    'catatan_kepala' => $penilaianGuru->catatan_kepala,
                    'total_nilai' => $penilaianGuru->total_nilai,
                    'rata_rata' => $penilaianGuru->rata_rata,
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Terjadi kesalahan: ' . $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ], 500);
        }
    }

    /**
     * Test AI analysis (for debugging)
     */
    public function testAI($id)
    {
        try {
            $user = Auth::user();
            
            $penilaianGuru = PenilaianGuru::with([
                'guru.lembaga',
                'kepala',
                'distribusiInstrumen.instrumen',
                'distribusiInstrumen.komponen',
                'jawabanGuru.instrumenItem.subKomponen.komponenPenilaian'
            ])
            ->where('kepala_id', $user->id)
            ->findOrFail($id);

            $aiService = new AIAnalysisService();
            
            return response()->json([
                'success' => true,
                'ai_enabled' => $aiService->isEnabled(),
                'settings' => [
                    'gemini_enabled' => $aiService->settings->gemini_enabled,
                    'gemini_api_key_set' => !empty($aiService->settings->gemini_api_key),
                    'gemini_model' => $aiService->settings->gemini_model,
                ],
                'test_analysis' => $aiService->generateKepalaAnalysis($penilaianGuru)
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'AI Test Error: ' . $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ], 500);
        }
    }

    /**
     * Format kabupaten name (remove KABUPATEN prefix)
     */
    private function formatKabupaten($kabupaten): string
    {
        if (!$kabupaten) {
            return 'PLACEHOLDER';
        }
        
        // Remove "KABUPATEN" prefix if exists
        $kabupaten = str_replace('KABUPATEN ', '', strtoupper($kabupaten));
        $kabupaten = str_replace('KOTA ', '', $kabupaten);
        
        return $kabupaten;
    }

    /**
     * Generate PDF raport for penilaian kepala (kepala melihat penilaian dirinya)
     */
    public function generateRaportPenilaianKepala($id)
    {
        try {
            $user = Auth::user();

            $penilaian = PenilaianKepala::with([
                'kepala.lembagaKepala',
                'pengawas',
                'distribusiInstrumen.instrumen',
                'distribusiInstrumen.komponen',
                'jawabanKepala.instrumenItem.subKomponen.komponenPenilaian',
            ])
                ->where('kepala_id', $user->id)
                ->findOrFail($id);

            if (!in_array($penilaian->status, ['validated', 'final'])) {
                return redirect()->back()->with('error', 'Raport hanya untuk penilaian berstatus validated atau final.');
            }

            $jawaban = $penilaian->jawabanKepala()
                ->leftJoin('instrumen_penilaians', 'jawaban_kepala.instrumen_item_id', '=', 'instrumen_penilaians.id')
                ->leftJoin('sub_komponens', 'instrumen_penilaians.sub_komponen_id', '=', 'sub_komponens.id')
                ->leftJoin('komponen_penilaians', 'sub_komponens.komponen_penilaian_id', '=', 'komponen_penilaians.id')
                ->with('instrumenItem')
                ->orderBy('komponen_penilaians.urutan', 'asc')
                ->orderBy('sub_komponens.urutan', 'asc')
                ->orderBy('instrumen_penilaians.no_urutan', 'asc')
                ->select('jawaban_kepala.*')
                ->get();

            // Group jawaban by sub komponen for analysis
            $subKomponenAnalysis = $this->prepareSubKomponenAnalysisPenilaianKepala($jawaban);

            // Load AI analysis from database
            $analisisPengawas = $penilaian->catatan_pengawas;
            
            // Fallback: Generate hanya jika benar-benar belum ada atau error
            if (!$analisisPengawas || 
                strpos($analisisPengawas, 'Terjadi kesalahan') !== false ||
                strpos($analisisPengawas, 'Analisis AI tidak diaktifkan') !== false ||
                strpos($analisisPengawas, 'Analisis tidak dapat dihasilkan') !== false) {
                
                \Log::warning('AI analysis not found in database, generating as fallback', [
                    'penilaian_id' => $penilaian->id,
                    'status' => $penilaian->status
                ]);
                
                try {
                    $aiService = new AIAnalysisService();
                    if ($aiService->isEnabled()) {
                        $analisisPengawas = $aiService->generatePengawasAnalysisKepala($penilaian);
                        // Update the database with new analysis
                        $penilaian->update(['catatan_pengawas' => $analisisPengawas]);
                    } else {
                        $analisisPengawas = 'Analisis AI tidak diaktifkan. Silakan hubungi administrator untuk mengaktifkan fitur AI.';
                    }
                } catch (\Exception $e) {
                    \Log::error('Pengawas AI Analysis Error (fallback): ' . $e->getMessage());
                    $analisisPengawas = 'Terjadi kesalahan dalam menghasilkan analisis. Silakan finalisasi ulang penilaian untuk generate analisis AI.';
                }
            }

            // Calculate total and average
            $totalNilai = $jawaban->sum('nilai_pengawas');
            $jumlahItem = $jawaban->whereNotNull('nilai_pengawas')->where('nilai_pengawas', '>', 0)->count();
            $rataRata = $jumlahItem > 0 ? round($totalNilai / $jumlahItem, 2) : 0;

            // Get lembaga - lembagaKepala() returns collection, so use first()
            $lembaga = $penilaian->kepala?->lembagaKepala?->first();
            
            // Debug: Log lembaga and logo info
            if ($lembaga) {
                \Log::info('Raport Lembaga Info (Kepala):', [
                    'lembaga_id' => $lembaga->id,
                    'lembaga_nama' => $lembaga->nama_lembaga,
                    'logo_lembaga' => $lembaga->logo_lembaga,
                    'logo_path_exists' => $lembaga->logo_lembaga ? file_exists(storage_path('app/public/' . $lembaga->logo_lembaga)) : false,
                ]);
            }

            // Get kabupaten from lembaga
            $kabupaten = '';
            if ($lembaga) {
                try {
                    $kabupaten = $lembaga->kabupaten_name;
                } catch (\Exception $e) {
                    $kabupaten = $lembaga->kabupaten ?? '';
                }
                
                if (empty($kabupaten)) {
                    $kabupaten = $lembaga->kabupaten ?? '';
                }
            }

            $data = [
                'penilaian' => $penilaian,
                'jawaban' => $jawaban,
                'subKomponenAnalysis' => $subKomponenAnalysis,
                'analisisPengawas' => $analisisPengawas,
                'kepala' => $penilaian->kepala,
                'pengawas' => $penilaian->pengawas,
                'lembaga' => $lembaga,
                'instrumen' => $penilaian->distribusiInstrumen?->instrumen,
                'komponen' => $penilaian->distribusiInstrumen?->komponen,
                'totalNilai' => $totalNilai,
                'rataRata' => $rataRata,
                'tanggal' => now()->format('d F Y'),
                'tahun' => now()->year,
                'kabupaten' => $this->formatKabupaten($kabupaten),
            ];

            $pdf = Pdf::loadView('pengawas.raport.template', $data);
            $pdf->setPaper('A4', 'portrait');

            $filename = 'Raport_Penilaian_Kepala_' . ($penilaian->kepala?->name ?? 'kepala') . '_' . now()->format('Y-m-d') . '.pdf';
            return $pdf->download($filename);
            
        } catch (\Exception $e) {
            \Log::error('PDF Generation Error (Kepala): ' . $e->getMessage());
            \Log::error('PDF Generation Stack Trace: ' . $e->getTraceAsString());
            
            return redirect()->back()
                ->with('error', 'Terjadi kesalahan saat menghasilkan PDF. Error: ' . $e->getMessage());
        }
    }

    /**
     * Prepare sub komponen analysis data for penilaian kepala
     */
    private function prepareSubKomponenAnalysisPenilaianKepala($jawaban)
    {
        // Group by sub_komponen_id using callback to ensure proper grouping
        $groupedBySubKomponen = $jawaban->groupBy(function($jawaban) {
            return $jawaban->instrumenItem?->sub_komponen_id ?? 'unknown';
        });
        
        $analysis = [];

        foreach ($groupedBySubKomponen as $subKomponenId => $jawabanGroup) {
            // Skip if sub_komponen_id is unknown or first item doesn't have instrumenItem
            if ($subKomponenId === 'unknown' || !$jawabanGroup->first()->instrumenItem) {
                continue;
            }
            
            $subKomponen = $jawabanGroup->first()->instrumenItem->subKomponen;
            
            // Filter nilai_pengawas yang valid (not null and > 0)
            $nilaiPengawas = $jawabanGroup->pluck('nilai_pengawas')->filter(function($v) {
                return $v !== null && $v !== '' && $v > 0;
            })->values();
            
            // Filter nilai_kepala (nilai_guru_self) yang valid (not null and > 0)
            $nilaiKepala = $jawabanGroup->pluck('nilai_guru_self')->filter(function($v) {
                return $v !== null && $v !== '' && $v > 0;
            })->values();
            
            // Calculate rata-rata hanya jika ada nilai pengawas yang valid
            if ($nilaiPengawas->isNotEmpty()) {
                $rataRataPengawas = $nilaiPengawas->avg();
                $kategori = $this->determineCategory($rataRataPengawas);
                
                // Calculate rata-rata kepala jika ada nilai yang valid
                $rataRataKepala = null;
                if ($nilaiKepala->isNotEmpty()) {
                    $rataRataKepala = $nilaiKepala->avg();
                }
                
                $analysis[] = [
                    'nama_sub_komponen' => $subKomponen->nama_sub_komponen ?? 'Unknown',
                    'deskripsi' => $subKomponen->deskripsi ?? '',
                    'rata_rata_pengawas' => round($rataRataPengawas, 2),
                    'rata_rata_kepala' => $rataRataKepala ? round($rataRataKepala, 2) : null,
                    'kategori' => $kategori,
                    'jumlah_item' => $jawabanGroup->count(),
                    'jumlah_terisi_pengawas' => $nilaiPengawas->count(),
                    'jumlah_terisi_kepala' => $nilaiKepala->count(),
                    'nilai_detail' => $nilaiPengawas->toArray(),
                    'indikator_kinerja' => $jawabanGroup->map(function($jawaban) {
                        return [
                            'indikator' => $jawaban->instrumenItem?->indikator_kinerja ?? 'Unknown',
                            'nilai_kepala' => $jawaban->nilai_guru_self ?? '-',
                            'nilai_pengawas' => $jawaban->nilai_pengawas ?? '-',
                            'bukti' => $jawaban->keterangan_bukti ?? '-',
                            'url_bukti' => $jawaban->url_bukti ?? '-',
                        ];
                    })->toArray()
                ];
            }
        }

        // Sort by sub_komponen urutan if available
        usort($analysis, function($a, $b) {
            // Try to maintain order from database
            return 0;
        });

        return $analysis;
    }

    /**
     * Determine category based on score
     */
    private function determineCategory(float $score): string
    {
        if ($score >= 4.5) return 'Sangat Baik';
        if ($score >= 3.5) return 'Baik';
        if ($score >= 2.5) return 'Cukup';
        return 'Kurang';
    }
}
