<?php

namespace App\Http\Controllers\Kepala;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class ProfileController extends Controller
{
    /**
     * Display the user's profile
     */
    public function index()
    {
        $user = Auth::user();
        return view('kepala.profile.index', compact('user'));
    }

    /**
     * Show the form for editing the user's profile
     */
    public function edit()
    {
        $user = Auth::user();
        return view('kepala.profile.edit', compact('user'));
    }

    /**
     * Update the user's profile
     */
    public function update(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'phone' => ['nullable', 'string', 'max:20'],
            'nip' => ['nullable', 'string', 'max:30'],
            'nuptk' => ['nullable', 'string', 'max:30'],
            'nama_lengkap' => ['required', 'string', 'max:255'],
            'gelar_depan' => ['nullable', 'string', 'max:50'],
            'gelar_belakang' => ['nullable', 'string', 'max:50'],
            'tempat_lahir' => ['required', 'string', 'max:100'],
            'tanggal_lahir' => ['required', 'date', 'before:today'],
            'jenis_kelamin' => ['required', 'in:L,P'],
            'pendidikan_terakhir' => ['required', 'string', 'max:100'],
            'jurusan' => ['nullable', 'string', 'max:100'],
            'alamat' => ['required', 'string'],
            'telepon' => ['nullable', 'string', 'max:20'],
            'foto_profil' => ['nullable', 'file', 'max:5120'],
        ]);

        try {
            // Debug request data
            \Log::info('Kepala Profile update request:', [
                'has_file' => $request->hasFile('foto_profil'),
                'file_valid' => $request->file('foto_profil') ? $request->file('foto_profil')->isValid() : false,
                'all_files' => $request->allFiles()
            ]);
            
            // Update user basic info
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
            ]);

            // Handle photo upload
            if ($request->hasFile('foto_profil')) {
                try {
                    $file = $request->file('foto_profil');
                    
                    // Debug file info
                    \Log::info('Kepala File upload info:', [
                        'original_name' => $file->getClientOriginalName(),
                        'mime_type' => $file->getMimeType(),
                        'size' => $file->getSize(),
                        'extension' => $file->getClientOriginalExtension()
                    ]);
                    
                    // Manual validation for image
                    $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
                    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
                    
                    $mimeType = $file->getMimeType();
                    $extension = strtolower($file->getClientOriginalExtension());
                    
                    if (!in_array($mimeType, $allowedMimes) && !in_array($extension, $allowedExtensions)) {
                        return back()->withInput()
                            ->with('error', 'Format file tidak didukung. Gunakan JPEG, PNG, JPG, GIF, atau WEBP. MIME: ' . $mimeType . ', Extension: ' . $extension);
                    }
                    
                    // Check file size (5MB = 5120KB)
                    if ($file->getSize() > 5120 * 1024) {
                        return back()->withInput()
                            ->with('error', 'Ukuran file terlalu besar. Maksimal 5MB.');
                    }
                    
                    // Delete old photo if exists
                    if ($user->foto_profil && Storage::disk('public')->exists($user->foto_profil)) {
                        Storage::disk('public')->delete($user->foto_profil);
                    }

                    // Store new photo
                    $photoPath = $file->store('profile-photos', 'public');
                    $user->update(['foto_profil' => $photoPath]);
                    
                    \Log::info('Kepala Photo uploaded successfully:', ['path' => $photoPath]);
                } catch (\Exception $e) {
                    \Log::error('Kepala Photo upload failed:', ['error' => $e->getMessage()]);
                    return back()->withInput()
                        ->with('error', 'Gagal mengupload foto: ' . $e->getMessage());
                }
            }

            // Update kepala profile if exists
            if ($user->kepala) {
                $user->kepala->update([
                    'nip' => $request->nip,
                    'nuptk' => $request->nuptk,
                    'nama_lengkap' => $request->nama_lengkap,
                    'gelar_depan' => $request->gelar_depan,
                    'gelar_belakang' => $request->gelar_belakang,
                    'tempat_lahir' => $request->tempat_lahir,
                    'tanggal_lahir' => $request->tanggal_lahir,
                    'jenis_kelamin' => $request->jenis_kelamin,
                    'pendidikan_terakhir' => $request->pendidikan_terakhir,
                    'jurusan' => $request->jurusan,
                    'alamat' => $request->alamat,
                    'telepon' => $request->telepon,
                ]);
            }

            return redirect()->route('kepala.profile.index')
                ->with('success', 'Profil berhasil diperbarui!');

        } catch (\Exception $e) {
            return back()->withInput()
                ->with('error', 'Terjadi kesalahan saat memperbarui profil: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for changing password
     */
    public function editPassword()
    {
        return view('kepala.profile.edit-password');
    }

    /**
     * Update the user's password
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);

        try {
            Auth::user()->update([
                'password' => Hash::make($request->password),
            ]);

            return redirect()->route('kepala.profile.index')
                ->with('success', 'Password berhasil diubah!');

        } catch (\Exception $e) {
            return back()->withInput()
                ->with('error', 'Terjadi kesalahan saat mengubah password: ' . $e->getMessage());
        }
    }
}
