<?php

namespace App\Http\Controllers\Kepala;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\PenilaianGuru;
use App\Models\DistribusiInstrumen;
use App\Models\JawabanGuru;
use App\Services\AIAnalysisService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PenilaianController extends Controller
{
    /**
     * Display a listing of teacher assessments for review
     */
    public function index()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();
        
        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        // Get all penilaian for this kepala
        $penilaianGuru = PenilaianGuru::with([
            'guru',
            'distribusiInstrumen.instrumen',
            'distribusiInstrumen.komponen'
        ])
            ->where('kepala_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        // Get statistics
        $stats = [
            'total_penilaian' => PenilaianGuru::where('kepala_id', $user->id)->count(),
            'draft' => PenilaianGuru::where('kepala_id', $user->id)->where('status', 'draft')->count(),
            'submitted' => PenilaianGuru::where('kepala_id', $user->id)->where('status', 'submitted')->count(),
            'revision' => PenilaianGuru::where('kepala_id', $user->id)->where('status', 'revision')->count(),
            'validated' => PenilaianGuru::where('kepala_id', $user->id)->where('status', 'validated')->count(),
            'final' => PenilaianGuru::where('kepala_id', $user->id)->where('status', 'final')->count(),
        ];

        return view('kepala.penilaian.index', compact('penilaianGuru', 'stats', 'lembaga'));
    }

    /**
     * Display the review page for teacher assessments
     */
    public function review()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();
        
        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        // Get submitted penilaian that need review
        $penilaianGuru = PenilaianGuru::with([
            'guru',
            'distribusiInstrumen.instrumen',
            'distribusiInstrumen.komponen',
            'jawabanGuru.instrumenItem'
        ])
            ->where('kepala_id', $user->id)
            ->where('status', 'submitted')
            ->orderBy('tanggal_submit', 'asc')
            ->paginate(10);

        // Get statistics
        $stats = [
            'pending_review' => PenilaianGuru::where('kepala_id', $user->id)->where('status', 'submitted')->count(),
            'total_reviewed' => PenilaianGuru::where('kepala_id', $user->id)->whereIn('status', ['validated', 'revision'])->count(),
        ];

        return view('kepala.penilaian.review', compact('penilaianGuru', 'stats', 'lembaga'));
    }

    /**
     * Show the form for reviewing a specific teacher assessment
     */
    public function show($id)
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();
        
        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun.');
        }

        // Get penilaian for this kepala
        $penilaianGuru = PenilaianGuru::with([
            'guru',
            'distribusiInstrumen.instrumen',
            'distribusiInstrumen.komponen',
            'jawabanGuru.instrumenItem'
        ])
            ->where('kepala_id', $user->id)
            ->findOrFail($id);

        // Get jawaban guru with instrumen items and sort by sub komponen
        $jawabanGuru = $penilaianGuru->jawabanGuru()
            ->leftJoin('instrumen_penilaians', 'jawaban_guru.instrumen_item_id', '=', 'instrumen_penilaians.id')
            ->leftJoin('sub_komponens', 'instrumen_penilaians.sub_komponen_id', '=', 'sub_komponens.id')
            ->leftJoin('komponen_penilaians', 'sub_komponens.komponen_penilaian_id', '=', 'komponen_penilaians.id')
            ->with('instrumenItem')
            ->orderBy('komponen_penilaians.urutan', 'asc')
            ->orderBy('sub_komponens.urutan', 'asc')
            ->orderBy('instrumen_penilaians.no_urutan', 'asc')
            ->select('jawaban_guru.*')
            ->get();

        return view('kepala.penilaian.show', compact('penilaianGuru', 'jawabanGuru'));
    }

    /**
     * Update the specified teacher assessment with review
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        
        // Get penilaian for this kepala first to validate against instrument data
        $penilaianGuru = PenilaianGuru::with('jawabanGuru.instrumenItem')
            ->where('kepala_id', $user->id)
            ->findOrFail($id);

        // Build dynamic validation rules based on instrument data
        $validationRules = [
            'jawaban' => 'required|array',
            'catatan_kepala' => 'nullable|string|max:1000',
            'status' => 'required|in:validated,revision'
        ];

        foreach ($penilaianGuru->jawabanGuru as $jawaban) {
            if ($jawaban->instrumenItem) {
                $validationRules["jawaban.{$jawaban->id}.nilai_kepala"] = "required|integer|min:{$jawaban->instrumenItem->skor_min}|max:{$jawaban->instrumenItem->skor_max}";
                $validationRules["jawaban.{$jawaban->id}.catatan_kepala"] = 'nullable|string|max:500';
            }
        }

        // Validate request
        $request->validate($validationRules);

        try {
            DB::beginTransaction();

            // Update jawaban guru
            foreach ($request->jawaban as $jawabanId => $data) {
                $jawaban = JawabanGuru::where('penilaian_guru_id', $penilaianGuru->id)
                    ->findOrFail($jawabanId);
                
                $jawaban->update([
                    'nilai_kepala' => $data['nilai_kepala'],
                    'catatan_kepala' => $data['catatan_kepala'] ?? null,
                ]);
            }

            // Calculate total nilai and rata-rata
            $totalNilai = $penilaianGuru->jawabanGuru()->sum('nilai_kepala');
            $jumlahItem = $penilaianGuru->jawabanGuru()->count();
            $rataRata = $jumlahItem > 0 ? $totalNilai / $jumlahItem : 0;

            // Generate AI analysis if status is validated or final
            $catatanKepala = $request->catatan_kepala;
            if (in_array($request->status, ['validated', 'final'])) {
                try {
                    $aiService = new AIAnalysisService();
                    $aiAnalysis = $aiService->generateKepalaAnalysis($penilaianGuru);
                    $catatanKepala = $aiAnalysis;
                } catch (\Exception $e) {
                    // Log error but continue with manual catatan
                    \Log::error('AI Analysis Error: ' . $e->getMessage());
                }
            }

            // Update penilaian guru
            $penilaianGuru->update([
                'status' => $request->status,
                'total_nilai' => $totalNilai,
                'rata_rata' => round($rataRata, 2),
                'catatan_kepala' => $catatanKepala,
                'tanggal_validasi' => now(),
            ]);

            DB::commit();

            $statusText = $request->status == 'validated' ? 'disetujui' : 'perlu direvisi';
            return redirect()->route('kepala.penilaian.review')
                ->with('success', "Penilaian guru berhasil direview dan {$statusText}.");

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Terjadi kesalahan saat menyimpan review. Silakan coba lagi.')
                ->withInput();
        }
    }

    /**
     * Get statistics for dashboard
     */
    public function getStats()
    {
        $user = Auth::user();
        
        $stats = [
            'pending_review' => PenilaianGuru::where('kepala_id', $user->id)->where('status', 'submitted')->count(),
            'total_reviewed' => PenilaianGuru::where('kepala_id', $user->id)->whereIn('status', ['validated', 'revision'])->count(),
            'validated' => PenilaianGuru::where('kepala_id', $user->id)->where('status', 'validated')->count(),
            'revision' => PenilaianGuru::where('kepala_id', $user->id)->where('status', 'revision')->count(),
        ];

        return response()->json($stats);
    }
}
