<?php

namespace App\Http\Controllers\Kepala;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Models\Lembaga;
use App\Models\Guru;
use App\Models\Instrumen;
use Barryvdh\DomPDF\Facade\Pdf;

class LembagaController extends Controller
{
    public function __construct()
    {
        $this->middleware('role:kepala');
    }

    /**
     * Display the lembaga information for the kepala
     */
    public function index()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->with('pengawas', 'guru')->first();

        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        return view('kepala.lembaga.index', compact('lembaga'));
    }

    /**
     * Show the form for editing the lembaga information
     */
    public function edit()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();

        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        return view('kepala.lembaga.edit', compact('lembaga'));
    }

    /**
     * Update the lembaga information
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();

        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        // Add logo validation to existing rules
        $validationRules = Lembaga::validationRulesComplete($lembaga->id);
        $validationRules['logo_lembaga'] = 'nullable|file|max:5120'; // 5MB max

        $request->validate($validationRules);

        $data = $request->all();
        
        // Remove nama_kepala from data since it's auto-synced from kepala relationship
        unset($data['nama_kepala']);
        
        // Handle logo upload
        if ($request->hasFile('logo_lembaga')) {
            try {
                $file = $request->file('logo_lembaga');
                
                // Debug file info
                \Log::info('Lembaga Logo upload info:', [
                    'original_name' => $file->getClientOriginalName(),
                    'mime_type' => $file->getMimeType(),
                    'size' => $file->getSize(),
                    'extension' => $file->getClientOriginalExtension()
                ]);
                
                // Manual validation for image
                $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
                $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
                
                $mimeType = $file->getMimeType();
                $extension = strtolower($file->getClientOriginalExtension());
                
                if (!in_array($mimeType, $allowedMimes) && !in_array($extension, $allowedExtensions)) {
                    return back()->withInput()
                        ->with('error', 'Format file tidak didukung. Gunakan JPEG, PNG, JPG, GIF, atau WEBP. MIME: ' . $mimeType . ', Extension: ' . $extension);
                }
                
                // Check file size (5MB = 5120KB)
                if ($file->getSize() > 5120 * 1024) {
                    return back()->withInput()
                        ->with('error', 'Ukuran file terlalu besar. Maksimal 5MB.');
                }
                
                // Delete old logo if exists
                if ($lembaga->logo_lembaga && Storage::disk('public')->exists($lembaga->logo_lembaga)) {
                    Storage::disk('public')->delete($lembaga->logo_lembaga);
                }

                // Store new logo
                $logoPath = $file->store('lembaga-logos', 'public');
                $data['logo_lembaga'] = $logoPath;
                
                \Log::info('Lembaga Logo uploaded successfully:', ['path' => $logoPath]);
            } catch (\Exception $e) {
                \Log::error('Lembaga Logo upload failed:', ['error' => $e->getMessage()]);
                return back()->withInput()
                    ->with('error', 'Gagal mengupload logo: ' . $e->getMessage());
            }
        }
        
        // Update lembaga information
        $lembaga->update($data);

        return redirect()->route('kepala.lembaga.index')
            ->with('success', 'Data lembaga berhasil diperbarui!');
    }

    /**
     * Display guru management
     */
    public function guru()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->with('guru')->first();

        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        $guru = $lembaga->guru()->with('user')->paginate(15);

        return view('kepala.lembaga.guru', compact('lembaga', 'guru'));
    }

    /**
     * Display instrumen for kepala
     */
    public function instrumen()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->with('pengawas')->first();

        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        // Get instrumen yang sudah didistribusikan ke lembaga ini
        $instrumen = Instrumen::whereHas('distribusiInstrumen', function($query) use ($lembaga) {
            $query->where('lembaga_id', $lembaga->id);
        })->with('komponenPenilaians')->paginate(15);

        return view('kepala.lembaga.instrumen', compact('lembaga', 'instrumen'));
    }

    /**
     * Generate PDF profil lembaga
     */
    public function downloadProfilPDF()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->with('pengawas', 'guru', 'kepala')->first();

        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        try {
            // Prepare logo for PDF
            $logoBase64 = null;
            if ($lembaga->logo_lembaga) {
                $logoPath = storage_path('app/public/' . $lembaga->logo_lembaga);
                if (!file_exists($logoPath)) {
                    $logoPath = public_path('storage/' . $lembaga->logo_lembaga);
                }
                if (!file_exists($logoPath)) {
                    $logoPath = base_path('storage/app/public/' . $lembaga->logo_lembaga);
                }
                if (file_exists($logoPath)) {
                    try {
                        $logoData = file_get_contents($logoPath);
                        $logoMime = mime_content_type($logoPath);
                        $logoBase64 = 'data:' . $logoMime . ';base64,' . base64_encode($logoData);
                        \Log::info('Logo loaded successfully for PDF', [
                            'lembaga_id' => $lembaga->id,
                            'logo_path' => $logoPath,
                            'logo_size' => strlen($logoData),
                            'mime_type' => $logoMime
                        ]);
                    } catch (\Exception $e) {
                        \Log::error('Error loading logo for PDF', [
                            'lembaga_id' => $lembaga->id,
                            'logo_path' => $logoPath,
                            'error' => $e->getMessage()
                        ]);
                    }
                } else {
                    \Log::warning('Logo file not found for PDF', [
                        'lembaga_id' => $lembaga->id,
                        'logo_lembaga' => $lembaga->logo_lembaga,
                        'tried_paths' => [
                            storage_path('app/public/' . $lembaga->logo_lembaga),
                            public_path('storage/' . $lembaga->logo_lembaga),
                            base_path('storage/app/public/' . $lembaga->logo_lembaga)
                        ]
                    ]);
                }
            } else {
                \Log::info('No logo set for lembaga', ['lembaga_id' => $lembaga->id]);
            }

            // Format kabupaten
            $kabupaten = $this->formatKabupaten($lembaga->kabupaten_name ?? $lembaga->kabupaten);

            $data = [
                'lembaga' => $lembaga,
                'logoBase64' => $logoBase64,
                'kabupaten' => $kabupaten,
                'tanggal' => now()->format('d F Y'),
            ];

            $pdf = Pdf::loadView('kepala.lembaga.profil-pdf', $data);
            $pdf->setPaper('A4', 'portrait');

            $filename = 'Profil_Lembaga_' . str_replace(' ', '_', $lembaga->nama_lembaga) . '_' . now()->format('Y-m-d') . '.pdf';
            return $pdf->download($filename);

        } catch (\Exception $e) {
            \Log::error('PDF Profil Lembaga Generation Error: ' . $e->getMessage());
            \Log::error('PDF Profil Lembaga Stack Trace: ' . $e->getTraceAsString());
            
            return redirect()->back()
                ->with('error', 'Terjadi kesalahan saat menghasilkan PDF. Error: ' . $e->getMessage());
        }
    }

    /**
     * Format kabupaten name
     */
    private function formatKabupaten($kabupaten)
    {
        if (empty($kabupaten)) {
            return '';
        }
        
        // Remove common prefixes
        $kabupaten = preg_replace('/^(KABUPATEN|KOTA|KAB\.|KOTA\.)\s*/i', '', $kabupaten);
        
        return ucwords(strtolower($kabupaten));
    }

    /**
     * Generate PDF cover raport penilaian kepala
     */
    public function downloadCoverRaport()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->with('kepala')->first();

        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        try {
            // Prepare logo for PDF
            $logoBase64 = null;
            if ($lembaga->logo_lembaga) {
                $logoPath = storage_path('app/public/' . $lembaga->logo_lembaga);
                if (!file_exists($logoPath)) {
                    $logoPath = public_path('storage/' . $lembaga->logo_lembaga);
                }
                if (!file_exists($logoPath)) {
                    $logoPath = base_path('storage/app/public/' . $lembaga->logo_lembaga);
                }
                if (file_exists($logoPath)) {
                    try {
                        $logoData = file_get_contents($logoPath);
                        $logoMime = mime_content_type($logoPath);
                        $logoBase64 = 'data:' . $logoMime . ';base64,' . base64_encode($logoData);
                    } catch (\Exception $e) {
                        \Log::error('Error loading logo for cover raport', [
                            'lembaga_id' => $lembaga->id,
                            'error' => $e->getMessage()
                        ]);
                    }
                }
            }

            // Get current year
            $tahun = now()->year;

            // Format alamat
            $alamat = $lembaga->alamat_lengkap;
            if ($lembaga->kelurahan_name || $lembaga->kelurahan) {
                $alamat .= ', ' . ($lembaga->kelurahan_name ?? $lembaga->kelurahan);
            }
            if ($lembaga->kecamatan_name || $lembaga->kecamatan) {
                $alamat .= ', ' . ($lembaga->kecamatan_name ?? $lembaga->kecamatan);
            }
            if ($lembaga->kabupaten_name || $lembaga->kabupaten) {
                $kabupaten = $this->formatKabupaten($lembaga->kabupaten_name ?? $lembaga->kabupaten);
                $alamat .= ', ' . $kabupaten;
            }
            if ($lembaga->provinsi_name || $lembaga->provinsi) {
                $alamat .= ', ' . ($lembaga->provinsi_name ?? $lembaga->provinsi);
            }
            if ($lembaga->kode_pos) {
                $alamat .= ' ' . $lembaga->kode_pos;
            }

            $data = [
                'lembaga' => $lembaga,
                'logoBase64' => $logoBase64,
                'tahun' => $tahun,
                'alamat' => $alamat,
            ];

            $pdf = Pdf::loadView('kepala.lembaga.cover-raport', $data);
            $pdf->setPaper('A4', 'portrait');

            $filename = 'Cover_Raport_Penilaian_Kepala_' . str_replace(' ', '_', $lembaga->nama_lembaga) . '_' . $tahun . '.pdf';
            return $pdf->download($filename);

        } catch (\Exception $e) {
            \Log::error('PDF Cover Raport Generation Error: ' . $e->getMessage());
            \Log::error('PDF Cover Raport Stack Trace: ' . $e->getTraceAsString());
            
            return redirect()->back()
                ->with('error', 'Terjadi kesalahan saat menghasilkan PDF. Error: ' . $e->getMessage());
        }
    }
}
