<?php

namespace App\Http\Controllers\Kepala;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Guru;
use App\Models\User;
use App\Models\PenilaianGuru;
use App\Models\DistribusiInstrumen;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class GuruController extends Controller
{
    /**
     * Display a listing of teachers in kepala's school
     */
    public function index()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();
        
        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        // Get all teachers in this lembaga with their latest penilaian status
        $guru = Guru::with(['user', 'penilaianGuru' => function($query) {
            $query->latest()->take(1);
        }])
        ->where('lembaga_id', $lembaga->id)
        ->orderBy('nama_lengkap')
        ->paginate(15);

        // Get statistics
        $stats = [
            'total_guru' => $guru->total(),
            'active_guru' => $guru->count(),
            'guru_with_penilaian' => Guru::where('lembaga_id', $lembaga->id)
                ->whereHas('penilaianGuru')
                ->count(),
            'pending_review' => PenilaianGuru::whereHas('guru', function($query) use ($lembaga) {
                $query->where('lembaga_id', $lembaga->id);
            })->where('status', 'submitted')->count(),
        ];

        // Get active instrument distributions for this lembaga
        $activeInstrumen = DistribusiInstrumen::with(['instrumen'])
            ->where('lembaga_id', $lembaga->id)
            ->where('status', 'active')
            ->get();

        return view('kepala.guru.index', compact('guru', 'stats', 'lembaga', 'activeInstrumen'));
    }

    /**
     * Display the specified teacher
     */
    public function show($id)
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();
        
        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun.');
        }

        // Get teacher details
        $guru = Guru::with(['user', 'lembaga'])
            ->where('lembaga_id', $lembaga->id)
            ->findOrFail($id);

        // Get all penilaian for this teacher
        $penilaianGuru = PenilaianGuru::with([
            'distribusiInstrumen.instrumen',
            'jawabanGuru.instrumenItem'
        ])
        ->where('guru_id', $guru->id)
        ->where('kepala_id', $user->id)
        ->orderBy('created_at', 'desc')
        ->get();

        // Get penilaian statistics
        $penilaianStats = [
            'total_penilaian' => $penilaianGuru->count(),
            'draft' => $penilaianGuru->where('status', 'draft')->count(),
            'submitted' => $penilaianGuru->where('status', 'submitted')->count(),
            'revision' => $penilaianGuru->where('status', 'revision')->count(),
            'validated' => $penilaianGuru->where('status', 'validated')->count(),
            'final' => $penilaianGuru->where('status', 'final')->count(),
        ];

        // Get latest penilaian
        $latestPenilaian = $penilaianGuru->first();

        return view('kepala.guru.show', compact('guru', 'penilaianGuru', 'penilaianStats', 'latestPenilaian'));
    }

    /**
     * Show the form for creating a new guru
     */
    public function create()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();
        
        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        return view('kepala.guru.create', compact('lembaga'));
    }

    /**
     * Store a newly created guru in storage
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();
        
        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun.');
        }

        // Validation rules for user + guru profile
        $request->validate([
            // User fields
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'phone' => 'nullable|string|max:20',
            
            // Guru profile fields
            'nip' => 'nullable|string|max:30',
            'nuptk' => 'nullable|string|max:30',
            'nama_lengkap' => 'required|string|max:255',
            'gelar_depan' => 'nullable|string|max:50',
            'gelar_belakang' => 'nullable|string|max:50',
            'tempat_lahir' => 'required|string|max:100',
            'tanggal_lahir' => 'required|date|before:today',
            'jenis_kelamin' => 'required|in:L,P',
            'pendidikan_terakhir' => 'required|string|max:100',
            'jurusan' => 'nullable|string|max:100',
            'mata_pelajaran' => 'required|string|max:100',
            'kelas_diampu' => 'required|string|max:100',
            'status_kepegawaian' => 'required|in:PNS,PPPK,GTY,GTT',
            'status_sertifikasi' => 'required|in:sudah,belum',
            'tahun_sertifikasi' => 'nullable|integer|min:1990|max:' . date('Y'),
            'no_sertifikat' => 'nullable|string|max:100',
            'status_inpassing' => 'required|in:sudah,belum',
            'golongan' => 'nullable|string|max:20',
            'tmt_golongan' => 'nullable|date',
            'alamat' => 'required|string',
            'telepon' => 'nullable|string|max:20',
        ]);

        DB::beginTransaction();
        try {
            // Create user account
            $userGuru = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'phone' => $request->phone,
                'is_active' => true,
            ]);

            // Assign guru role
            $userGuru->assignRole('guru');

            // Create guru profile
            $guru = Guru::create([
                'user_id' => $userGuru->id,
                'lembaga_id' => $lembaga->id,
                'nip' => $request->nip,
                'nuptk' => $request->nuptk,
                'nama_lengkap' => $request->nama_lengkap,
                'gelar_depan' => $request->gelar_depan,
                'gelar_belakang' => $request->gelar_belakang,
                'tempat_lahir' => $request->tempat_lahir,
                'tanggal_lahir' => $request->tanggal_lahir,
                'jenis_kelamin' => $request->jenis_kelamin,
                'pendidikan_terakhir' => $request->pendidikan_terakhir,
                'jurusan' => $request->jurusan,
                'mata_pelajaran' => $request->mata_pelajaran,
                'kelas_diampu' => $request->kelas_diampu,
                'status_kepegawaian' => $request->status_kepegawaian,
                'status_sertifikasi' => $request->status_sertifikasi,
                'tahun_sertifikasi' => $request->tahun_sertifikasi,
                'no_sertifikat' => $request->no_sertifikat,
                'status_inpassing' => $request->status_inpassing,
                'golongan' => $request->golongan,
                'tmt_golongan' => $request->tmt_golongan,
                'alamat' => $request->alamat,
                'telepon' => $request->telepon,
            ]);

            DB::commit();

            return redirect()->route('kepala.guru.index')
                ->with('success', 'Guru berhasil dibuat! Akun guru: ' . $request->email);

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()
                ->with('error', 'Terjadi kesalahan saat membuat guru: ' . $e->getMessage());
        }
    }
}
