<?php

namespace App\Http\Controllers\Kepala;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\DistribusiInstrumen;
use App\Models\PenilaianKepala;
use App\Models\InstrumenPenilaian;
use App\Models\JawabanKepala;
use App\Models\Notification;
use Illuminate\Support\Facades\Auth;

class InstrumenController extends Controller
{
    /**
     * Display a listing of distributed instruments for kepala's school
     */
    public function index()
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();
        
        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun. Silakan hubungi admin.');
        }

        // Get distributed instruments for this lembaga
        $distribusi = DistribusiInstrumen::with([
            'instrumen', 
            'komponen',
            'penilaianKepala'
        ])
            ->where('lembaga_id', $lembaga->id)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        // Get statistics
        $stats = [
            'total_instrumen' => $distribusi->total(),
            'active_instrumen' => DistribusiInstrumen::where('lembaga_id', $lembaga->id)
                ->where('status', 'active')->count(),
            'completed_penilaian' => DistribusiInstrumen::where('lembaga_id', $lembaga->id)
                ->whereHas('penilaianKepala', function($query) use ($user) {
                    $query->where('kepala_id', $user->id)->whereIn('status', ['validated', 'final']);
                })->count(),
            'pending_review' => PenilaianKepala::where('kepala_id', $user->id)
                ->where('status', 'submitted')->count(),
        ];

        // Get notifications for this kepala
        $notifications = Notification::with('related')
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Get unread notification count
        $unreadCount = Notification::where('user_id', $user->id)
            ->where('is_read', false)
            ->count();

        return view('kepala.instrumen.index', compact('distribusi', 'stats', 'lembaga', 'notifications', 'unreadCount'));
    }

    /**
     * Display the specified instrument distribution
     */
    public function show($id)
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();
        
        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun.');
        }

        // Get distribution for this lembaga
        $distribusi = DistribusiInstrumen::with([
            'instrumen.komponenPenilaians.subKomponens.instrumenPenilaians', 
            'komponen.instrumenPenilaians',
            'penilaianKepala'
        ])
            ->where('lembaga_id', $lembaga->id)
            ->findOrFail($id);

        // Get penilaian for this kepala
        $penilaianGuru = $distribusi->penilaianKepala()
            ->where('kepala_id', $user->id)
            ->with(['jawabanKepala.instrumenItem'])
            ->get();

        // Get statistics
        $penilaianStats = [
            'total_kepala' => $penilaianGuru->count(),
            'total' => $penilaianGuru->count(),
            'draft' => $penilaianGuru->where('status', 'draft')->count(),
            'submitted' => $penilaianGuru->where('status', 'submitted')->count(),
            'revision' => $penilaianGuru->where('status', 'revision')->count(),
            'validated' => $penilaianGuru->where('status', 'validated')->count() + $penilaianGuru->where('status', 'final')->count(),
        ];

        return view('kepala.instrumen.show', compact('distribusi', 'penilaianGuru', 'penilaianStats'));
    }

    /**
     * Mark notification as read
     */
    public function markNotificationAsRead(Request $request, $notificationId)
    {
        $user = Auth::user();
        
        $notification = Notification::where('id', $notificationId)
            ->where('user_id', $user->id)
            ->firstOrFail();

        $notification->markAsRead();

        return response()->json(['success' => true]);
    }

    /**
     * Mark all notifications as read
     */
    public function markAllNotificationsAsRead(Request $request)
    {
        $user = Auth::user();
        
        Notification::where('user_id', $user->id)
            ->where('is_read', false)
            ->update([
                'is_read' => true,
                'read_at' => now()
            ]);

        return response()->json(['success' => true]);
    }

    /**
     * Get notifications (AJAX)
     */
    public function getNotifications(Request $request)
    {
        $user = Auth::user();
        
        $notifications = Notification::with('related')
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get();

        $unreadCount = Notification::where('user_id', $user->id)
            ->where('is_read', false)
            ->count();

        return response()->json([
            'notifications' => $notifications,
            'unread_count' => $unreadCount
        ]);
    }

    /**
     * Form pengisian instrumen oleh kepala (self-assessment)
     */
    public function isi($id, Request $request)
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->first();
        if (!$lembaga) {
            return redirect()->route('kepala.dashboard')
                ->with('error', 'Anda belum diassign ke lembaga manapun.');
        }

        $distribusi = DistribusiInstrumen::with(['instrumen.komponenPenilaians.subKomponens', 'komponen'])
            ->where('lembaga_id', $lembaga->id)
            ->findOrFail($id);

        // Ambil daftar item instrumen sesuai jenis distribusi
        if ($distribusi->komponen) {
            // Stabilize ordering: some records share the same no_urutan
            $itemsQuery = $distribusi->komponen->instrumenPenilaians()
                ->orderBy('no_urutan')
                ->orderBy('id');
        } else {
            $subIds = $distribusi->instrumen->komponenPenilaians->flatMap(function ($k) {
                return $k->subKomponens->pluck('id');
            })->values();
            $itemsQuery = InstrumenPenilaian::whereIn('sub_komponen_id', $subIds)
                ->orderBy('no_urutan')
                ->orderBy('id');
        }

        $perPage = (int)($request->get('per_page', 10));
        $items = $itemsQuery->paginate($perPage)->withQueryString();

        // PenilaianKepala record untuk kepala ini
        $penilaian = PenilaianKepala::firstOrCreate([
            'distribusi_instrumen_id' => $distribusi->id,
            'kepala_id' => $user->id,
        ], [
            'status' => 'draft',
        ]);

        // Jawaban existing keyed by item id
        $jawabanMap = JawabanKepala::where('penilaian_kepala_id', $penilaian->id)
            ->get()->keyBy('instrumen_item_id');

        return view('kepala.instrumen.isi', compact('distribusi', 'items', 'penilaian', 'jawabanMap', 'lembaga'));
    }

    /**
     * Simpan pengisian instrumen oleh kepala (self-assessment)
     */
    public function simpanIsi($id, Request $request)
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->firstOrFail();
        $distribusi = DistribusiInstrumen::where('lembaga_id', $lembaga->id)->findOrFail($id);
        $penilaian = PenilaianKepala::firstOrCreate([
            'distribusi_instrumen_id' => $distribusi->id,
            'kepala_id' => $user->id,
        ], [ 'status' => 'draft' ]);

        // Block edit ketika sudah divalidasi/final
        if (in_array($penilaian->status, ['validated', 'final'])) {
            return back()->with('error', 'Penilaian sudah divalidasi. Perubahan tidak diizinkan.');
        }

        $dataJawaban = $request->get('jawaban', []);
        foreach ($dataJawaban as $itemId => $val) {
            JawabanKepala::updateOrCreate([
                'penilaian_kepala_id' => $penilaian->id,
                'instrumen_item_id' => $itemId,
            ], [
                'nilai_guru_self' => $val['nilai_guru_self'] ?? null,
                'url_bukti' => $val['url_bukti'] ?? null,
                'keterangan_bukti' => $val['keterangan_bukti'] ?? null,
            ]);
        }

        // Opsi update status bila diminta submit
        if ($request->get('submit_status') === 'submitted') {
            // Validasi: semua item pada distribusi harus terisi nilai dan URL bukti
            if ($distribusi->komponen) {
                $allItemIds = $distribusi->komponen->instrumenPenilaians()->pluck('id')->toArray();
            } else {
                $subIds = $distribusi->instrumen->komponenPenilaians->flatMap(function ($k) {
                    return $k->subKomponens->pluck('id');
                })->values();
                $allItemIds = InstrumenPenilaian::whereIn('sub_komponen_id', $subIds)->pluck('id')->toArray();
            }

            $totalItems = count($allItemIds);
            $filled = JawabanKepala::where('penilaian_kepala_id', $penilaian->id)
                ->whereIn('instrumen_item_id', $allItemIds)
                ->whereNotNull('nilai_guru_self')
                ->where('nilai_guru_self', '>', 0)
                ->whereNotNull('url_bukti')
                ->where('url_bukti', '!=', '')
                ->count();

            if ($filled < $totalItems) {
                return back()->with('error', 'Tidak dapat submit: semua item harus diisi nilai dan URL bukti (' . $filled . '/' . $totalItems . ').');
            }

            $penilaian->update([
                'status' => 'submitted',
                'tanggal_submit' => now(),
            ]);
        }

        return back()->with('success', 'Isian instrumen berhasil disimpan.');
    }

    /**
     * Ubah status penilaian kembali ke draft agar bisa diedit
     */
    public function ubahStatus($id, Request $request)
    {
        $user = Auth::user();
        $lembaga = $user->lembagaKepala()->firstOrFail();
        $distribusi = DistribusiInstrumen::where('lembaga_id', $lembaga->id)->findOrFail($id);
        
        $penilaian = PenilaianKepala::where('distribusi_instrumen_id', $distribusi->id)
            ->where('kepala_id', $user->id)
            ->firstOrFail();

        $newStatus = $request->input('status', 'draft');
        
        // Validasi status yang diizinkan
        $allowedStatuses = ['draft', 'revision'];
        if (!in_array($newStatus, $allowedStatuses)) {
            return back()->with('error', 'Status tidak valid. Hanya boleh diubah ke draft atau revision.');
        }

        // Ubah status dan reset tanggal submit/validasi jika perlu
        $updateData = [
            'status' => $newStatus,
        ];

        // Jika mengubah ke draft/revision, reset tanggal submit dan validasi
        if (in_array($newStatus, ['draft', 'revision'])) {
            $updateData['tanggal_submit'] = null;
            $updateData['tanggal_validasi'] = null;
        }

        $penilaian->update($updateData);

        $statusText = $newStatus == 'draft' ? 'draft' : 'perlu direvisi';
        return back()->with('success', 'Status penilaian berhasil diubah menjadi ' . $statusText . '. Anda sekarang dapat mengedit isian instrumen.');
    }
}
