<?php

namespace App\Http\Controllers\Guru;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\DistribusiInstrumen;
use App\Models\PenilaianGuru;
use App\Models\Notification;
use Illuminate\Support\Facades\Auth;

class InstrumenController extends Controller
{
    /**
     * Display a listing of instruments available for the guru
     */
    public function index()
    {
        $user = Auth::user();
        $guru = $user->guru;
        
        if (!$guru) {
            return redirect()->route('guru.dashboard')
                ->with('error', 'Profil guru tidak ditemukan. Silakan hubungi admin.');
        }

        // Get instruments distributed to this guru's lembaga
        $distribusi = DistribusiInstrumen::with(['instrumen.komponenPenilaians.subKomponens', 'komponen.subKomponens'])
            ->where('lembaga_id', $guru->lembaga_id)
            ->where('status', 'active')
            ->orderBy('deadline', 'asc')
            ->get();

        // Get penilaian for this guru
        $penilaianGuru = PenilaianGuru::with([
            'distribusiInstrumen.instrumen',
            'distribusiInstrumen.komponen'
        ])
            ->where('guru_id', $guru->id)
            ->orderBy('created_at', 'desc')
            ->get();

        // Statistics
        $stats = [
            'total_instrumen' => $distribusi->count(),
            'penilaian_by_status' => [
                'draft' => $penilaianGuru->where('status', 'draft')->count(),
                'submitted' => $penilaianGuru->where('status', 'submitted')->count(),
                'revision' => $penilaianGuru->where('status', 'revision')->count(),
                'validated' => $penilaianGuru->where('status', 'validated')->count(),
                'final' => $penilaianGuru->where('status', 'final')->count(),
            ]
        ];

        return view('guru.instrumen.index', compact('distribusi', 'penilaianGuru', 'stats', 'guru'));
    }

    /**
     * Show the form for filling an instrument
     */
    public function isi()
    {
        $user = Auth::user();
        $guru = $user->guru;
        
        if (!$guru) {
            return redirect()->route('guru.dashboard')
                ->with('error', 'Profil guru tidak ditemukan.');
        }

        // Get instruments that can be filled (draft or revision status)
        $penilaianGuru = PenilaianGuru::with([
            'distribusiInstrumen.komponen.subKomponens.instrumenPenilaians',
            'distribusiInstrumen.instrumen',
            'jawabanGuru'
        ])
            ->where('guru_id', $guru->id)
            ->whereIn('status', ['draft', 'revision'])
            ->orderBy('created_at', 'desc')
            ->get();

        return view('guru.instrumen.isi', compact('penilaianGuru', 'guru'));
    }

    /**
     * Show instrument structure
     */
    public function struktur($id)
    {
        $user = Auth::user();
        $guru = $user->guru;
        
        if (!$guru) {
            return redirect()->route('guru.dashboard')
                ->with('error', 'Profil guru tidak ditemukan.');
        }

        // Get specific penilaian
        $penilaianGuru = PenilaianGuru::with([
            'distribusiInstrumen.komponen.subKomponens.instrumenPenilaians',
            'distribusiInstrumen.instrumen.komponenPenilaians.subKomponens.instrumenPenilaians',
            'jawabanGuru'
        ])
        ->where('guru_id', $guru->id)
        ->findOrFail($id);

        // Get all items for structure display
        $allItems = $this->getAllItems($penilaianGuru);
        
        // Calculate progress
        $jawabanByItem = $penilaianGuru->jawabanGuru->keyBy('instrumen_item_id');
        $progress = $allItems->map(function($item) use ($jawabanByItem) {
            $jawaban = $jawabanByItem->get($item['id']);
            $isComplete = $jawaban && $jawaban->url_bukti && $jawaban->keterangan_bukti;
            $isPartial = $jawaban && ($jawaban->url_bukti || $jawaban->keterangan_bukti);
            
            return [
                'id' => $item['id'],
                'status' => $isComplete ? 'complete' : ($isPartial ? 'partial' : 'empty')
            ];
        });

        return view('guru.instrumen.struktur', compact(
            'penilaianGuru', 
            'allItems',
            'progress'
        ));
    }

    /**
     * Show specific instrument to fill
     */
    public function show($id)
    {
        
        $user = Auth::user();
        $guru = $user->guru;
        
        if (!$guru) {
            return redirect()->route('guru.dashboard')
                ->with('error', 'Profil guru tidak ditemukan.');
        }

        // Get specific penilaian
        $penilaianGuru = PenilaianGuru::with([
            'distribusiInstrumen.komponen.subKomponens.instrumenPenilaians',
            'distribusiInstrumen.instrumen',
            'jawabanGuru'
        ])
        ->where('guru_id', $guru->id)
        ->findOrFail($id);

        // Check if this penilaian can be edited
        if (!in_array($penilaianGuru->status, ['draft', 'revision'])) {
            return redirect()->route('guru.instrumen.index')
                ->with('error', 'Instrumen ini sudah tidak dapat diedit.');
        }

        // Get instrument items only for the distributed komponen
        $allItems = collect();
        
        if ($penilaianGuru->distribusiInstrumen->komponen) {
            // Per komponen distribution - only show items from the distributed komponen
            $komponen = $penilaianGuru->distribusiInstrumen->komponen;
            $subKomponenIndex = 0;
            foreach ($komponen->subKomponens as $subKomponen) {
                $subKomponenIndex++;
                foreach ($subKomponen->instrumenPenilaians as $item) {
                    $urutan = $subKomponenIndex * 1000 + $item->no_urutan;
                    $allItems->push([
                        'id' => $item->id,
                        'komponen' => $komponen->nama_komponen,
                        'sub_komponen' => $subKomponen->nama_sub_komponen,
                        'indikator' => $item->indikator_kinerja,
                        'evidensi' => $item->evidensi_kinerja,
                        'skor_min' => $item->skor_min,
                        'skor_max' => $item->skor_max,
                        'no_urutan' => $item->no_urutan,
                        'urutan' => $urutan
                    ]);
                }
            }
        } else {
            // Semua komponen distribution - show all items from all komponen
            $instrumen = $penilaianGuru->distribusiInstrumen->instrumen;
            $subKomponenIndex = 0;
            foreach ($instrumen->komponenPenilaians as $komponen) {
                foreach ($komponen->subKomponens as $subKomponen) {
                    $subKomponenIndex++;
                    foreach ($subKomponen->instrumenPenilaians as $item) {
                        $urutan = $subKomponenIndex * 1000 + $item->no_urutan;
                        $allItems->push([
                            'id' => $item->id,
                            'komponen' => $komponen->nama_komponen,
                            'sub_komponen' => $subKomponen->nama_sub_komponen,
                            'indikator' => $item->indikator_kinerja,
                            'evidensi' => $item->evidensi_kinerja,
                            'skor_min' => $item->skor_min,
                            'skor_max' => $item->skor_max,
                            'no_urutan' => $item->no_urutan,
                            'urutan' => $urutan
                        ]);
                    }
                }
            }
        }

        // Sort by urutan
        $allItems = $allItems->sortBy('urutan')->values();

        // Get current item (default to first if no item_id provided)
        $itemId = request('item_id', $allItems->first()['id'] ?? null);
        
        // Check if this is validation page
        if ($itemId === 'validation') {
            $currentItem = null; // No current item for validation page
        } else {
            $currentItem = $allItems->firstWhere('id', $itemId);
        }
        
        if (!$currentItem && $itemId !== 'validation') {
            return redirect()->route('guru.instrumen.index')
                ->with('error', 'Item instrumen tidak ditemukan.');
        }

        // Get existing answer for current item
        $existingAnswer = $penilaianGuru->jawabanGuru->firstWhere('instrumen_item_id', $itemId);

        // Calculate progress for navigation - guru hanya perlu mengisi bukti
        $jawabanByItem = $penilaianGuru->jawabanGuru->keyBy('instrumen_item_id');
        $progress = $allItems->map(function($item) use ($jawabanByItem, $currentItem) {
            $jawaban = $jawabanByItem->get($item['id']);
            $isComplete = $jawaban && $jawaban->url_bukti && $jawaban->keterangan_bukti;
            $isPartial = $jawaban && ($jawaban->url_bukti || $jawaban->keterangan_bukti);
            
            // Check if currentItem exists and has id
            $isCurrent = false;
            if ($currentItem && isset($currentItem['id'])) {
                $isCurrent = $item['id'] == $currentItem['id'];
            }
            
            return [
                'id' => $item['id'],
                'status' => $isComplete ? 'complete' : ($isPartial ? 'partial' : 'empty'),
                'current' => $isCurrent
            ];
        });

        return view('guru.instrumen.show', compact(
            'penilaianGuru', 
            'guru', 
            'allItems', 
            'currentItem', 
            'existingAnswer',
            'progress'
        ));
    }

    /**
     * Store or update single item answer
     */
    public function store(Request $request, $id)
    {
        
        $user = Auth::user();
        $guru = $user->guru;
        
        if (!$guru) {
            \Log::error('Guru not found');
            return redirect()->route('guru.dashboard')
                ->with('error', 'Profil guru tidak ditemukan.');
        }

        $penilaianGuru = PenilaianGuru::where('guru_id', $guru->id)
            ->findOrFail($id);
            

        // Check if this penilaian can be edited
        if (!in_array($penilaianGuru->status, ['draft', 'revision'])) {
            return redirect()->route('guru.instrumen.index')
                ->with('error', 'Instrumen ini sudah tidak dapat diedit.');
        }

        // Check if this is a final submit
        if ($request->has('final_submit') && $request->final_submit) {
            return $this->finalSubmit($request, $penilaianGuru);
        }

        // Get the item to validate score range
        $item = \App\Models\InstrumenPenilaian::findOrFail($request->item_id);
        
        // Single item validation - guru hanya mengisi bukti, tidak menilai diri
        $request->validate([
            'item_id' => 'required|integer',
            'url_bukti' => 'required|url|max:500',
            'keterangan_bukti' => 'required|string|max:1000',
        ]);

        try {
            // Save single answer - guru hanya mengisi bukti, tidak menilai diri
            \App\Models\JawabanGuru::updateOrCreate([
                'penilaian_guru_id' => $penilaianGuru->id,
                'instrumen_item_id' => $request->item_id,
            ], [
                'url_bukti' => $request->url_bukti,
                'keterangan_bukti' => $request->keterangan_bukti,
                // nilai_guru_self tidak diisi oleh guru, akan diisi oleh kepala sekolah saat review
            ]);

            // Get next item or redirect to completion
            $allItems = $this->getAllItems($penilaianGuru);
            $currentIndex = $allItems->search(function($item) use ($request) {
                return $item['id'] == $request->item_id;
            });

            $nextIndex = $currentIndex + 1;
            
            if ($nextIndex < $allItems->count()) {
                // Go to next item
                $nextItem = $allItems[$nextIndex];
                return redirect()->route('guru.instrumen.show', [$id, 'item_id' => $nextItem['id']])
                    ->with('success', 'Jawaban berhasil disimpan!');
            } else {
                // All items completed, go to validation page
                return redirect()->route('guru.instrumen.show', [$id, 'item_id' => 'validation'])
                    ->with('success', 'Semua item telah diisi! Silakan review dan submit.');
            }

        } catch (\Exception $e) {
            return back()->withInput()
                ->with('error', 'Terjadi kesalahan saat menyimpan jawaban: ' . $e->getMessage());
        }
    }

    /**
     * Final submit all answers
     */
    private function finalSubmit($request, $penilaianGuru)
    {
        
        try {
            // Update penilaian status to submitted
            $penilaianGuru->update([
                'status' => 'submitted',
                'tanggal_submit' => now(),
            ]);
            
            // Create notification for kepala sekolah
            $kepalaId = $penilaianGuru->kepala_id;
            if ($kepalaId) {
                Notification::create([
                    'type' => 'instrumen_submitted',
                    'title' => 'Instrumen Baru Dikirim',
                    'message' => 'Guru ' . $penilaianGuru->guru->nama . ' telah mengirim instrumen penilaian yang perlu direview.',
                    'user_id' => $kepalaId,
                    'related_id' => $penilaianGuru->id,
                    'related_type' => 'App\Models\PenilaianGuru',
                    'is_read' => false
                ]);
            }
            
            return redirect()->route('guru.instrumen.isi')
                ->with('success', 'Instrumen berhasil disubmit ke kepala sekolah!');

        } catch (\Exception $e) {
            
            return back()->withInput()
                ->with('error', 'Terjadi kesalahan saat submit instrumen: ' . $e->getMessage());
        }
    }

    /**
     * Get all items for navigation
     */
    private function getAllItems($penilaianGuru)
    {
        $allItems = collect();
        
        if ($penilaianGuru->distribusiInstrumen->komponen) {
            // Per komponen distribution - only show items from the distributed komponen
            $komponen = $penilaianGuru->distribusiInstrumen->komponen;
            $subKomponenIndex = 0;
            foreach ($komponen->subKomponens as $subKomponen) {
                $subKomponenIndex++;
                foreach ($subKomponen->instrumenPenilaians as $item) {
                    $allItems->push([
                        'id' => $item->id,
                        'komponen' => $komponen->nama_komponen,
                        'sub_komponen' => $subKomponen->nama_sub_komponen,
                        'indikator' => $item->indikator_kinerja,
                        'evidensi' => $item->evidensi_kinerja,
                        'skor_min' => $item->skor_min,
                        'skor_max' => $item->skor_max,
                        'no_urutan' => $item->no_urutan,
                        'urutan' => $subKomponenIndex * 1000 + $item->no_urutan // Urut berdasarkan sub komponen
                    ]);
                }
            }
        } else {
            // Semua komponen distribution - show all items from all komponen
            $komponenIndex = 0;
            foreach ($penilaianGuru->distribusiInstrumen->instrumen->komponenPenilaians as $komponen) {
                $komponenIndex++;
                $subKomponenIndex = 0;
                foreach ($komponen->subKomponens as $subKomponen) {
                    $subKomponenIndex++;
                    foreach ($subKomponen->instrumenPenilaians as $item) {
                        $allItems->push([
                            'id' => $item->id,
                            'komponen' => $komponen->nama_komponen,
                            'sub_komponen' => $subKomponen->nama_sub_komponen,
                            'indikator' => $item->indikator_kinerja,
                            'evidensi' => $item->evidensi_kinerja,
                            'skor_min' => $item->skor_min,
                            'skor_max' => $item->skor_max,
                            'no_urutan' => $item->no_urutan,
                            'urutan' => $komponenIndex * 10000 + $subKomponenIndex * 1000 + $item->no_urutan // Urut berdasarkan komponen dan sub komponen
                        ]);
                    }
                }
            }
        }
        
        return $allItems->sortBy('urutan')->values();
    }

    /**
     * Mark notification as read
     */
    public function markNotificationAsRead(Request $request, $notificationId)
    {
        $user = Auth::user();
        
        $notification = Notification::where('id', $notificationId)
            ->where('user_id', $user->id)
            ->firstOrFail();

        $notification->markAsRead();

        return response()->json(['success' => true]);
    }

    /**
     * Mark all notifications as read
     */
    public function markAllNotificationsAsRead(Request $request)
    {
        $user = Auth::user();
        
        Notification::where('user_id', $user->id)
            ->where('is_read', false)
            ->update([
                'is_read' => true,
                'read_at' => now()
            ]);

        return response()->json(['success' => true]);
    }

    /**
     * Get notifications (AJAX)
     */
    public function getNotifications(Request $request)
    {
        $user = Auth::user();
        
        $notifications = Notification::with('related')
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get();

        $unreadCount = Notification::where('user_id', $user->id)
            ->where('is_read', false)
            ->count();

        return response()->json([
            'notifications' => $notifications,
            'unread_count' => $unreadCount
        ]);
    }
}
