<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class WilayahController extends Controller
{
    private $regionsData;

    public function __construct()
    {
        // Load regions data from JSON file
        $jsonPath = storage_path('app/indonesia-regions.json');
        if (file_exists($jsonPath)) {
            $this->regionsData = json_decode(file_get_contents($jsonPath), true);
        } else {
            $this->regionsData = ['provinces' => [], 'regencies' => [], 'districts' => [], 'villages' => []];
        }
    }

    /**
     * Get all provinces
     */
    public function getProvinces()
    {
        return response()->json($this->regionsData['provinces'] ?? []);
    }

    /**
     * Get regencies by province ID
     */
    public function getRegencies(Request $request)
    {
        $provinceId = $request->get('province_id');
        
        if (!$provinceId) {
            return response()->json([]);
        }

        $regencies = $this->regionsData['regencies'][$provinceId] ?? [];
        return response()->json($regencies);
    }

    /**
     * Get districts by regency ID
     */
    public function getDistricts(Request $request)
    {
        $regencyId = $request->get('regency_id');
        
        if (!$regencyId) {
            return response()->json([]);
        }

        $districts = $this->regionsData['districts'][$regencyId] ?? [];
        return response()->json($districts);
    }

    /**
     * Get villages by district ID
     */
    public function getVillages(Request $request)
    {
        $districtId = $request->get('district_id');
        
        if (!$districtId) {
            return response()->json([]);
        }

        $villages = $this->regionsData['villages'][$districtId] ?? [];
        return response()->json($villages);
    }

    /**
     * Get province name by ID
     */
    public function getProvinceName($id)
    {
        $province = collect($this->regionsData['provinces'])->firstWhere('id', $id);
        return response()->json([
            'id' => $id,
            'name' => $province['name'] ?? $id
        ]);
    }

    /**
     * Get regency name by ID
     */
    public function getRegencyName(Request $request, $id)
    {
        $provinceId = $request->get('province_id');
        
        if ($provinceId) {
            $regencies = $this->regionsData['regencies'][$provinceId] ?? [];
            $regency = collect($regencies)->firstWhere('id', $id);
        } else {
            $regency = null;
            // Search in all regencies if no province specified
            foreach ($this->regionsData['regencies'] as $provinceRegencies) {
                $regency = collect($provinceRegencies)->firstWhere('id', $id);
                if ($regency) break;
            }
        }

        return response()->json([
            'id' => $id,
            'name' => $regency['name'] ?? $id
        ]);
    }

    /**
     * Get district name by ID
     */
    public function getDistrictName(Request $request, $id)
    {
        $regencyId = $request->get('regency_id');
        
        if ($regencyId) {
            $districts = $this->regionsData['districts'][$regencyId] ?? [];
            $district = collect($districts)->firstWhere('id', $id);
        } else {
            $district = null;
            // Search in all districts if no regency specified
            foreach ($this->regionsData['districts'] as $regencyDistricts) {
                $district = collect($regencyDistricts)->firstWhere('id', $id);
                if ($district) break;
            }
        }

        return response()->json([
            'id' => $id,
            'name' => $district['name'] ?? $id
        ]);
    }

    /**
     * Get village name by ID
     */
    public function getVillageName(Request $request, $id)
    {
        $districtId = $request->get('district_id');
        
        if ($districtId) {
            $villages = $this->regionsData['villages'][$districtId] ?? [];
            $village = collect($villages)->firstWhere('id', $id);
        } else {
            $village = null;
            // Search in all villages if no district specified
            foreach ($this->regionsData['villages'] as $districtVillages) {
                $village = collect($districtVillages)->firstWhere('id', $id);
                if ($village) break;
            }
        }

        return response()->json([
            'id' => $id,
            'name' => $village['name'] ?? $id
        ]);
    }
}