<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Lembaga;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;
use App\Exports\UsersExport;
use App\Exports\UsersTemplateExport;
use App\Imports\UsersImport;
use Maatwebsite\Excel\Facades\Excel;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('role:admin');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = User::with('roles');

        // Filter berdasarkan role
        if ($request->filled('role')) {
            $query->role($request->role);
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('is_active', $request->status == 'active' ? 1 : 0);
        }

        // Filter berdasarkan tanggal
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        // Sort
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $users = $query->paginate(15)->appends($request->query());
        $roles = Role::all();

        // Statistics
        $stats = [
            'total' => User::count(),
            'active' => User::where('is_active', true)->count(),
            'inactive' => User::where('is_active', false)->count(),
            'new_today' => User::whereDate('created_at', today())->count(),
            'by_role' => [
                'admin' => User::role('admin')->count(),
                'pengawas' => User::role('pengawas')->count(),
                'kepala' => User::role('kepala')->count(),
            ]
        ];

        return view('admin.users.index', compact('users', 'roles', 'stats'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $roles = Role::all();
        $lembaga = Lembaga::all();
        return view('admin.users.create', compact('roles', 'lembaga'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'phone' => 'nullable|string|max:20',
            'role' => 'required|exists:roles,name',
            'is_active' => 'boolean',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'phone' => $request->phone,
            'is_active' => $request->boolean('is_active', true),
        ]);

        $user->assignRole($request->role);

        return redirect()->route('admin.users.index')
            ->with('success', 'User berhasil dibuat!');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        $user->load(['roles', 'lembaga']);
        return view('admin.users.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $roles = Role::all();
        $lembaga = Lembaga::all();
        return view('admin.users.edit', compact('user', 'roles', 'lembaga'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:8|confirmed',
            'phone' => 'nullable|string|max:20',
            'role' => 'required|exists:roles,name',
            'is_active' => 'boolean',
        ]);

        $data = [
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'is_active' => $request->boolean('is_active', true),
        ];

        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $user->update($data);
        
        // Update role
        $user->syncRoles([$request->role]);

        return redirect()->route('admin.users.index')
            ->with('success', 'User berhasil diupdate!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // Jangan hapus user jika ini adalah admin terakhir
        if ($user->hasRole('admin') && User::role('admin')->count() <= 1) {
            return redirect()->route('admin.users.index')
                ->with('error', 'Tidak dapat menghapus admin terakhir!');
        }

        $user->delete();

        return redirect()->route('admin.users.index')
            ->with('success', 'User berhasil dihapus!');
    }

    /**
     * Toggle user status
     */
    public function toggleStatus(User $user)
    {
        $user->update(['is_active' => !$user->is_active]);
        
        $status = $user->is_active ? 'diaktifkan' : 'dinonaktifkan';
        return redirect()->back()
            ->with('success', "User berhasil {$status}!");
    }

    /**
     * Bulk actions for users
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:activate,deactivate,delete',
            'user_ids' => 'required|array',
            'user_ids.*' => 'exists:users,id'
        ]);

        $userIds = $request->user_ids;
        $action = $request->action;
        $count = 0;

        switch ($action) {
            case 'activate':
                $count = User::whereIn('id', $userIds)->update(['is_active' => true]);
                $message = "{$count} user berhasil diaktifkan!";
                break;
            case 'deactivate':
                $count = User::whereIn('id', $userIds)->update(['is_active' => false]);
                $message = "{$count} user berhasil dinonaktifkan!";
                break;
            case 'delete':
                // Jangan hapus admin terakhir
                $adminUsers = User::whereIn('id', $userIds)->role('admin')->get();
                if ($adminUsers->count() > 0 && User::role('admin')->count() <= $adminUsers->count()) {
                    return redirect()->back()
                        ->with('error', 'Tidak dapat menghapus semua admin!');
                }
                $count = User::whereIn('id', $userIds)->delete();
                $message = "{$count} user berhasil dihapus!";
                break;
        }

        return redirect()->back()->with('success', $message);
    }

    /**
     * Export users to Excel
     */
    public function export(Request $request)
    {
        $filters = $request->only(['role', 'status', 'date_from', 'date_to', 'search']);
        $fileName = 'users_export_' . date('Y-m-d_H-i-s') . '.xlsx';
        
        return Excel::download(new UsersExport($filters), $fileName);
    }

    /**
     * Download template Excel for import
     */
    public function downloadTemplate()
    {
        try {
            $fileName = 'template_import_users_' . date('Y-m-d') . '.xlsx';
            
            return Excel::download(new UsersTemplateExport(), $fileName);
        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show import form
     */
    public function showImport()
    {
        return view('admin.users.import');
    }

    /**
     * Import users from Excel
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:2048'
        ]);

        try {
            $import = new UsersImport();
            Excel::import($import, $request->file('file'));
            
            $results = $import->getResults();
            
            $message = "Import berhasil! {$results['imported_count']} user berhasil diimport.";
            if (!empty($results['errors'])) {
                $message .= " Beberapa error: " . implode(', ', array_slice($results['errors'], 0, 3));
                if (count($results['errors']) > 3) {
                    $message .= " dan " . (count($results['errors']) - 3) . " error lainnya.";
                }
            }
            
            return redirect()->route('admin.users.index')
                ->with('success', $message);
                
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Terjadi kesalahan saat import: ' . $e->getMessage());
        }
    }
}
