<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\KomponenPenilaian;
use App\Models\SubKomponen;
use Illuminate\Http\Request;

class SubKomponenController extends Controller
{
    public function __construct()
    {
        $this->middleware('role:admin');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(KomponenPenilaian $komponen)
    {
        $subKomponens = $komponen->subKomponens()->orderBy('urutan')->get();
        return view('admin.sub-komponen.index', compact('komponen', 'subKomponens'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(KomponenPenilaian $komponen)
    {
        return view('admin.sub-komponen.create', compact('komponen'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, KomponenPenilaian $komponen)
    {
        \Log::info('=== SUB KOMPONEN STORE DEBUG ===');
        \Log::info('Request data:', $request->all());
        \Log::info('Komponen ID:', ['id' => $komponen->id]);

        // Simple validation
        $request->validate([
            'nama_sub_komponen' => 'required|string|max:255',
            'urutan' => 'required|integer|min:1',
        ]);

        $data = [
            'komponen_penilaian_id' => $komponen->id,
            'nama_sub_komponen' => $request->nama_sub_komponen,
            'deskripsi' => $request->deskripsi,
            'urutan' => $request->urutan,
            'is_active' => $request->has('is_active') ? 1 : 0,
        ];

        \Log::info('Data to create:', $data);

        $subKomponen = SubKomponen::create($data);

        \Log::info('Sub komponen created:', $subKomponen->toArray());

        return redirect()->route('admin.sub-komponen.index', $komponen)
            ->with('success', 'Sub komponen berhasil dibuat!');
    }

    /**
     * Display the specified resource.
     */
    public function show(KomponenPenilaian $komponen, SubKomponen $subKomponen)
    {
        $subKomponen->load('instrumenPenilaians');
        return view('admin.sub-komponen.show', compact('komponen', 'subKomponen'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(KomponenPenilaian $komponen, SubKomponen $subKomponen)
    {
        return view('admin.sub-komponen.edit', compact('komponen', 'subKomponen'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, KomponenPenilaian $komponen, SubKomponen $subKomponen)
    {
        $request->validate(SubKomponen::validationRules($subKomponen->id));

        $subKomponen->update($request->all());

        return redirect()->route('admin.sub-komponen.index', $komponen)
            ->with('success', 'Sub komponen berhasil diupdate!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(KomponenPenilaian $komponen, SubKomponen $subKomponen)
    {
        // Cek apakah sudah ada instrumen penilaian
        if ($subKomponen->instrumenPenilaians()->count() > 0) {
            return redirect()->route('admin.sub-komponen.index', $komponen)
                ->with('error', 'Tidak dapat menghapus sub komponen yang sudah memiliki instrumen penilaian!');
        }

        $subKomponen->delete();

        return redirect()->route('admin.sub-komponen.index', $komponen)
            ->with('success', 'Sub komponen berhasil dihapus!');
    }
}
