<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AppSettings;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules\Password;

class SettingsController extends Controller
{
    /**
     * Display settings dashboard
     */
    public function index()
    {
        $settings = AppSettings::getSettings();
        $user = Auth::user();
        
        return view('admin.settings.index', compact('settings', 'user'));
    }

    /**
     * Update profile
     */
    public function updateProfile(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . Auth::id(),
            'phone' => 'nullable|string|max:20',
            'avatar' => 'nullable|file|mimetypes:image/png,image/x-png,image/jpeg,image/jpg,image/gif,image/webp|max:4096'
        ]);

        $user = Auth::user();
        $data = $request->only(['name', 'email', 'phone']);

        // Handle avatar upload
        if ($request->hasFile('avatar')) {
            // Delete old avatar if exists
            if ($user->avatar && Storage::disk('public')->exists($user->avatar)) {
                Storage::disk('public')->delete($user->avatar);
            }
            
            $avatarPath = $request->file('avatar')->store('avatars', 'public');
            $data['avatar'] = $avatarPath;
        }

        $user->update($data);

        return redirect()->route('admin.settings.index')
            ->with('success', 'Profil berhasil diperbarui!');
    }

    /**
     * Update password
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'password' => ['required', 'confirmed', Password::defaults()],
        ]);

        $user = Auth::user();

        // Check current password
        if (!Hash::check($request->current_password, $user->password)) {
            return redirect()->back()
                ->withErrors(['current_password' => 'Password saat ini tidak sesuai.']);
        }

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return redirect()->route('admin.settings.index')
            ->with('success', 'Password berhasil diperbarui!');
    }

    /**
     * Update app settings
     */
    public function updateAppSettings(Request $request)
    {
        $request->validate([
            'app_name' => 'required|string|max:255',
            'app_description' => 'nullable|string',
            'app_version' => 'required|string|max:20',
            'app_author' => 'nullable|string|max:255',
            'app_email' => 'nullable|email|max:255',
            'app_phone' => 'nullable|string|max:20',
            'app_address' => 'nullable|string',
            'app_website' => 'nullable|url|max:255',
            'app_timezone' => 'required|string',
            'app_locale' => 'required|string',
            'app_pagination' => 'required|integer|min:5|max:100',
            'app_logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'app_favicon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:1024',
        ]);

        $data = $request->except(['app_logo', 'app_favicon']);

        // Handle logo upload
        if ($request->hasFile('app_logo')) {
            $settings = AppSettings::getSettings();
            
            // Delete old logo if exists
            if ($settings->app_logo && Storage::disk('public')->exists($settings->app_logo)) {
                Storage::disk('public')->delete($settings->app_logo);
            }
            
            $logoPath = $request->file('app_logo')->store('app', 'public');
            $data['app_logo'] = $logoPath;
        }

        // Handle favicon upload
        if ($request->hasFile('app_favicon')) {
            $settings = AppSettings::getSettings();
            
            // Delete old favicon if exists
            if ($settings->app_favicon && Storage::disk('public')->exists($settings->app_favicon)) {
                Storage::disk('public')->delete($settings->app_favicon);
            }
            
            $faviconPath = $request->file('app_favicon')->store('app', 'public');
            $data['app_favicon'] = $faviconPath;
        }

        AppSettings::updateSettings($data);

        return redirect()->route('admin.settings.index')
            ->with('success', 'Pengaturan aplikasi berhasil diperbarui!');
    }

    /**
     * Update system settings
     */
    public function updateSystemSettings(Request $request)
    {
        $request->validate([
            'app_maintenance' => 'boolean',
            'maintenance_message' => 'nullable|string',
            'app_notifications' => 'boolean',
            'app_logs' => 'boolean',
            'backup_frequency' => 'required|in:daily,weekly,monthly',
            'backup_retention' => 'required|integer|min:1|max:365',
        ]);

        $data = $request->only([
            'app_maintenance',
            'maintenance_message',
            'app_notifications',
            'app_logs',
            'backup_frequency',
            'backup_retention'
        ]);

        AppSettings::updateSettings($data);

        return redirect()->route('admin.settings.index')
            ->with('success', 'Pengaturan sistem berhasil diperbarui!');
    }

    /**
     * Delete logo
     */
    public function deleteLogo()
    {
        $settings = AppSettings::getSettings();
        
        if ($settings->app_logo && Storage::disk('public')->exists($settings->app_logo)) {
            Storage::disk('public')->delete($settings->app_logo);
            $settings->update(['app_logo' => null]);
        }

        return redirect()->route('admin.settings.index')
            ->with('success', 'Logo berhasil dihapus!');
    }

    /**
     * Delete favicon
     */
    public function deleteFavicon()
    {
        $settings = AppSettings::getSettings();
        
        if ($settings->app_favicon && Storage::disk('public')->exists($settings->app_favicon)) {
            Storage::disk('public')->delete($settings->app_favicon);
            $settings->update(['app_favicon' => null]);
        }

        return redirect()->route('admin.settings.index')
            ->with('success', 'Favicon berhasil dihapus!');
    }

    /**
     * Update Gemini AI settings
     */
    public function updateGeminiSettings(Request $request)
    {
        $request->validate([
            'gemini_api_key' => 'nullable|string|max:255',
            'gemini_model' => 'required|string|in:gemini-2.5-flash,gemini-2.5-pro,gemini-1.5-flash-8b,gemini-1.5-flash,gemini-1.5-pro,gemini-pro,gemini-pro-vision',
            'gemini_enabled' => 'boolean',
            'gemini_max_tokens' => 'required|integer|min:1|max:8192',
            'gemini_temperature' => 'required|numeric|min:0|max:2',
        ]);

        $data = $request->only([
            'gemini_api_key',
            'gemini_model',
            'gemini_enabled',
            'gemini_max_tokens',
            'gemini_temperature'
        ]);

        // Convert boolean values
        $data['gemini_enabled'] = $request->has('gemini_enabled');

        AppSettings::updateSettings($data);

        return redirect()->route('admin.settings.index')
            ->with('success', 'Pengaturan Gemini AI berhasil diperbarui!');
    }
}