<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Lembaga;
use App\Models\User;
use Illuminate\Http\Request;

class LembagaController extends Controller
{
    public function __construct()
    {
        $this->middleware('role:admin');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Lembaga::with('pengawas', 'kepala', 'guru');

        // Filter berdasarkan jenis
        if ($request->filled('jenis')) {
            $query->where('jenis', $request->jenis);
        }

        // Filter berdasarkan pengawas
        if ($request->filled('pengawas_id')) {
            if ($request->pengawas_id === 'unassigned') {
                $query->whereNull('pengawas_id');
            } else {
                $query->where('pengawas_id', $request->pengawas_id);
            }
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_lembaga', 'like', "%{$search}%")
                  ->orWhere('npsn_nsm', 'like', "%{$search}%")
                  ->orWhere('nama_kepala', 'like', "%{$search}%");
            });
        }

        $lembaga = $query->paginate(15);
        $pengawas = User::role('pengawas')->get();
        $jenisLembaga = ['RA', 'MI', 'MTs', 'MA'];

        return view('admin.lembaga.index', compact('lembaga', 'pengawas', 'jenisLembaga'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.lembaga.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate(Lembaga::validationRulesAdmin());

        // Normalize jenis to uppercase
        $data = $request->all();
        $data['jenis'] = strtoupper($data['jenis']);
        
        // Set default values for other fields
        $data['pengawas_id'] = null;
        $data['kepala_id'] = null;
        $data['alamat_lengkap'] = '';
        $data['kelurahan'] = '';
        $data['kecamatan'] = '';
        $data['kabupaten'] = '';
        $data['provinsi'] = '';
        $data['kode_pos'] = '';
        $data['telepon'] = '';
        $data['email'] = '';
        $data['website'] = '';
        $data['nama_kepala'] = '';
        $data['nip_kepala'] = '';
        $data['akreditasi'] = null;
        $data['tahun_berdiri'] = null;

        Lembaga::create($data);

        return redirect()->route('admin.lembaga.index')
            ->with('success', 'Lembaga berhasil dibuat! Data lengkap akan dilengkapi nanti oleh kepala lembaga.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Lembaga $lembaga)
    {
        $lembaga->load(['pengawas', 'guru.user', 'distribusiInstrumen.instrumen']);
        return view('admin.lembaga.show', compact('lembaga'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Lembaga $lembaga)
    {
        $pengawas = User::role('pengawas')->get();
        return view('admin.lembaga.edit', compact('lembaga', 'pengawas'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Lembaga $lembaga)
    {
        $request->validate(Lembaga::validationRulesAdminEdit($lembaga->id));

        $data = $request->all();
        
        // Convert empty strings to null for regional fields
        $data['kelurahan'] = empty($data['kelurahan']) ? null : $data['kelurahan'];
        $data['kecamatan'] = empty($data['kecamatan']) ? null : $data['kecamatan'];
        $data['kabupaten'] = empty($data['kabupaten']) ? null : $data['kabupaten'];
        $data['provinsi'] = empty($data['provinsi']) ? null : $data['provinsi'];
        
        // Convert empty strings to null for kepala fields (since they're read-only)
        $data['nama_kepala'] = empty($data['nama_kepala']) ? null : $data['nama_kepala'];
        $data['nip_kepala'] = empty($data['nip_kepala']) ? null : $data['nip_kepala'];

        $lembaga->update($data);

        return redirect()->route('admin.lembaga.index')
            ->with('success', 'Lembaga berhasil diupdate!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Lembaga $lembaga)
    {
        // Cek apakah lembaga sudah memiliki pengawas
        if ($lembaga->pengawas_id !== null) {
            return redirect()->route('admin.lembaga.index')
                ->with('error', 'Tidak dapat menghapus lembaga yang sudah memiliki pengawas!');
        }

        // Cek apakah ada guru yang terdaftar
        if ($lembaga->guru()->count() > 0) {
            return redirect()->route('admin.lembaga.index')
                ->with('error', 'Tidak dapat menghapus lembaga yang masih memiliki guru!');
        }

        $lembaga->delete();

        return redirect()->route('admin.lembaga.index')
            ->with('success', 'Lembaga berhasil dihapus!');
    }

    /**
     * Assign kepala to lembaga
     */
    public function assignKepala(Request $request)
    {
        $request->validate([
            'lembaga_id' => 'required|exists:lembaga,id',
            'kepala_id' => 'required|exists:users,id',
        ]);

        // Check if there are available kepala
        $availableKepala = User::role('kepala')->whereDoesntHave('lembagaKepala')->get();
        if ($availableKepala->count() == 0) {
            return redirect()->back()
                ->with('error', 'Tidak ada kepala yang tersedia untuk diassign. Semua kepala sudah memiliki lembaga.');
        }

        $lembaga = Lembaga::findOrFail($request->lembaga_id);
        $kepala = User::findOrFail($request->kepala_id);

        // Check if kepala has kepala role
        if (!$kepala->hasRole('kepala')) {
            return redirect()->back()
                ->with('error', 'User yang dipilih bukan kepala lembaga!');
        }

        // Check if lembaga already has a kepala (1 lembaga = 1 kepala)
        if ($lembaga->kepala_id && $lembaga->kepala_id != $kepala->id) {
            $currentKepala = $lembaga->kepala;
            return redirect()->back()
                ->with('error', "Lembaga '{$lembaga->nama_lembaga}' sudah memiliki kepala: '{$currentKepala->name}'. Silakan ubah kepala yang ada terlebih dahulu.");
        }

        // Check if kepala is already assigned to another lembaga (should not happen with filtered dropdown)
        $existingAssignment = Lembaga::where('kepala_id', $kepala->id)->where('id', '!=', $lembaga->id)->first();
        if ($existingAssignment) {
            // This should not happen since we filter the dropdown, but just in case
            return redirect()->back()
                ->with('error', "Kepala '{$kepala->name}' sudah diassign ke lembaga '{$existingAssignment->nama_lembaga}'. Silakan pilih kepala lain.");
        }

        // Assign kepala to lembaga and sync data
        $lembaga->update([
            'kepala_id' => $kepala->id,
            'nama_kepala' => $kepala->name,
            'nip_kepala' => $kepala->nip ?? $lembaga->nip_kepala, // Keep existing if no NIP in user
        ]);

        return redirect()->route('admin.lembaga.index')
            ->with('success', "Kepala '{$kepala->name}' berhasil diassign ke lembaga '{$lembaga->nama_lembaga}'! Data kepala telah disinkronkan otomatis.");
    }
}
