<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Exports\InstrumenPenilaianTemplateExport;
use App\Imports\InstrumenPenilaianImport;
use App\Models\KomponenPenilaian;
use App\Models\SubKomponen;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;

class InstrumenPenilaianImportController extends Controller
{
    /**
     * Download template Excel untuk import instrumen penilaian
     */
    public function template(KomponenPenilaian $komponen, SubKomponen $subKomponen)
    {
        return Excel::download(
            new InstrumenPenilaianTemplateExport($komponen, $subKomponen),
            'template_instrumen_penilaian_' . $subKomponen->nama_sub_komponen . '.xlsx',
            \Maatwebsite\Excel\Excel::XLSX,
            [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => 'attachment; filename="template_instrumen_penilaian_' . $subKomponen->nama_sub_komponen . '.xlsx"'
            ]
        );
    }

    /**
     * Import data instrumen penilaian dari Excel
     */
    public function import(Request $request, KomponenPenilaian $komponen, SubKomponen $subKomponen)
    {
        $request->validate([
            'file' => ['required','mimes:xlsx','max:20480'],
        ]);

        try {
            $import = new InstrumenPenilaianImport($komponen, $subKomponen);
            Excel::import($import, $request->file('file'));

            $importedCount = $import->getRowCount();
            $failedCount = $import->failures()->count();

            $message = "Import selesai. Berhasil: {$importedCount} instrumen penilaian";

            if ($failedCount > 0) {
                $message .= ", Gagal: {$failedCount} baris";
                $errorDetails = [];
                foreach ($import->failures() as $failure) {
                    $errorDetails[] = "Baris {$failure->row()}: " . implode(', ', $failure->errors());
                }
                $message .= ". Detail: " . implode('; ', $errorDetails);
            }

            return redirect()->route('admin.instrumen-penilaian.index', ['komponen' => $komponen, 'subKomponen' => $subKomponen])->with('status', $message);

        } catch (\Throwable $e) {
            return back()->with('error', 'Import gagal: '.$e->getMessage());
        }
    }
}