<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Instrumen;
use App\Models\DistribusiInstrumen;
use App\Models\User;
use App\Exports\InstrumenHierarkisExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class InstrumenController extends Controller
{
    public function __construct()
    {
        $this->middleware('role:admin');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Instrumen::with('instrumenItems');

        // Filter berdasarkan status
        if ($request->filled('is_active')) {
            $query->where('is_active', $request->boolean('is_active'));
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('judul', 'like', "%{$search}%")
                  ->orWhere('tahun_ajaran', 'like', "%{$search}%");
            });
        }

        $instrumen = $query->paginate(15);

        return view('admin.instrumen.index', compact('instrumen'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.instrumen.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate(Instrumen::validationRules());

        Instrumen::create($request->all());

        return redirect()->route('admin.instrumen.index')
            ->with('success', 'Instrumen berhasil dibuat!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Instrumen $instrumen)
    {
        $instrumen->load([
            'komponenPenilaians.subKomponens.instrumenPenilaians',
            'distribusiInstrumen.lembaga'
        ]);
        return view('admin.instrumen.show', compact('instrumen'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Instrumen $instrumen)
    {
        $instrumen->load(['distribusiInstrumen.lembaga']);
        return view('admin.instrumen.edit', compact('instrumen'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Instrumen $instrumen)
    {
        $request->validate(Instrumen::validationRules($instrumen->id));

        $instrumen->update($request->all());

        return redirect()->route('admin.instrumen.index')
            ->with('success', 'Instrumen berhasil diupdate!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Instrumen $instrumen)
    {
        // Cek apakah sudah ada distribusi
        if ($instrumen->distribusiInstrumen()->count() > 0) {
            return redirect()->route('admin.instrumen.index')
                ->with('error', 'Tidak dapat menghapus instrumen yang sudah didistribusikan!');
        }

        $instrumen->delete();

        return redirect()->route('admin.instrumen.index')
            ->with('success', 'Instrumen berhasil dihapus!');
    }

    /**
     * Toggle status instrumen
     */
    public function toggleStatus(Instrumen $instrumen)
    {
        $instrumen->update(['is_active' => !$instrumen->is_active]);
        
        $status = $instrumen->is_active ? 'diaktifkan' : 'dinonaktifkan';
        return redirect()->back()
            ->with('success', "Instrumen berhasil {$status}!");
    }

    /**
     * Update deadline for all distributions of this instrument
     */
    public function updateDeadline(Request $request, Instrumen $instrumen)
    {
        $request->validate([
            'deadline' => 'required|date|after:today',
            'update_all' => 'nullable|boolean',
            'distribusi_ids' => 'nullable|array',
            'distribusi_ids.*' => 'exists:distribusi_instrumen,id',
        ]);

        $deadline = $request->deadline;
        $updateAll = $request->boolean('update_all', false);
        $distribusiIds = $request->distribusi_ids ?? [];

        DB::beginTransaction();
        try {
            if ($updateAll) {
                // Update semua distribusi aktif
                $instrumen->distribusiInstrumen()
                    ->where('status', 'active')
                    ->update(['deadline' => $deadline]);
                
                $count = $instrumen->distribusiInstrumen()
                    ->where('status', 'active')
                    ->count();
            } else {
                // Update distribusi yang dipilih saja
                if (empty($distribusiIds)) {
                    return back()->with('error', 'Pilih minimal satu distribusi untuk diupdate!');
                }
                
                DistribusiInstrumen::whereIn('id', $distribusiIds)
                    ->where('instrumen_id', $instrumen->id)
                    ->update(['deadline' => $deadline]);
                
                $count = count($distribusiIds);
            }

            DB::commit();
            
            return redirect()->route('admin.instrumen.edit', $instrumen)
                ->with('success', "Deadline berhasil diperpanjang untuk {$count} distribusi!");
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Gagal memperpanjang deadline: ' . $e->getMessage());
        }
    }

    /**
     * Export instrumen to Excel with hierarchical structure
     */
    public function exportExcel(Instrumen $instrumen)
    {
        $filename = 'Instrumen_' . str_replace(' ', '_', $instrumen->judul) . '_' . date('Y-m-d') . '.xlsx';
        
        return Excel::download(new InstrumenHierarkisExport($instrumen), $filename);
    }
}
