<?php

namespace App\Exports;

use App\Models\Instrumen;
use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithTitle;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class InstrumenHierarkisExport implements FromArray, WithHeadings, WithStyles, WithColumnWidths, WithTitle
{
    protected $instrumen;

    public function __construct(Instrumen $instrumen)
    {
        $this->instrumen = $instrumen;
    }

    /**
     * @return array
     */
    public function array(): array
    {
        $data = [];
        $komponenLetters = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J'];
        
        // Load data with relationships
        $this->instrumen->load([
            'komponenPenilaians.subKomponens.instrumenPenilaians'
        ]);

        foreach ($this->instrumen->komponenPenilaians()->orderBy('urutan')->get() as $komponenIndex => $komponen) {
            // Komponen row
            $data[] = [
                $komponenLetters[$komponenIndex],
                $komponen->nama_komponen,
                $komponen->deskripsi ?: 'Komponen Penilaian',
                '',
                '',
                'Komponen'
            ];

            foreach ($komponen->subKomponens()->orderBy('urutan')->get() as $subKomponen) {
                // Sub Komponen row
                $data[] = [
                    '',
                    $subKomponen->nama_sub_komponen,
                    $subKomponen->deskripsi ?: 'Sub Komponen',
                    '',
                    '',
                    'Sub Komponen'
                ];

                foreach ($subKomponen->instrumenPenilaians()->orderBy('no_urutan')->get() as $instrumenItem) {
                    // Instrumen row
                    $data[] = [
                        $komponenLetters[$komponenIndex] . $instrumenItem->no_urutan,
                        $instrumenItem->indikator_kinerja,
                        $instrumenItem->evidensi_kinerja,
                        $instrumenItem->skor_min,
                        $instrumenItem->skor_max,
                        'Instrumen'
                    ];
                }
            }
        }

        return $data;
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'No',
            'Indikator Kerja',
            'Evidensi Kerja',
            'Skor Min',
            'Skor Max',
            'Jenis'
        ];
    }

    /**
     * @return array
     */
    public function styles(Worksheet $sheet)
    {
        $styles = [];
        
        // Header row
        $styles[1] = [
            'font' => ['bold' => true, 'size' => 12],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => ['rgb' => '2E7D32']
            ],
            'alignment' => ['horizontal' => 'center', 'vertical' => 'center']
        ];

        // Title row (if exists)
        if ($sheet->getCell('A1')->getValue()) {
            $styles['A1:F1'] = [
                'font' => ['bold' => true, 'size' => 14],
                'alignment' => ['horizontal' => 'center'],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '1976D2']
                ]
            ];
        }

        // Style rows based on type
        $row = 2; // Start from row 2 (after header)
        $komponenLetters = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J'];
        
        foreach ($this->instrumen->komponenPenilaians()->orderBy('urutan')->get() as $komponenIndex => $komponen) {
            // Komponen row styling
            $styles[$row] = [
                'font' => ['bold' => true, 'size' => 11],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'E3F2FD']
                ],
                'alignment' => ['vertical' => 'top']
            ];
            $row++;

            foreach ($komponen->subKomponens()->orderBy('urutan')->get() as $subKomponen) {
                // Sub Komponen row styling
                $styles[$row] = [
                    'font' => ['bold' => true, 'size' => 10],
                    'fill' => [
                        'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                        'startColor' => ['rgb' => 'FFF3E0']
                    ],
                    'alignment' => ['vertical' => 'top']
                ];
                $row++;

                foreach ($subKomponen->instrumenPenilaians()->orderBy('no_urutan')->get() as $instrumenItem) {
                    // Instrumen row styling
                    $styles[$row] = [
                        'font' => ['size' => 10],
                        'fill' => [
                            'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                            'startColor' => ['rgb' => 'F5F5F5']
                        ],
                        'alignment' => ['vertical' => 'top', 'wrapText' => true]
                    ];
                    $row++;
                }
            }
        }

        return $styles;
    }

    /**
     * @return array
     */
    public function columnWidths(): array
    {
        return [
            'A' => 8,   // No
            'B' => 50,  // Indikator Kerja
            'C' => 40,  // Evidensi Kerja
            'D' => 10,  // Skor Min
            'E' => 10,  // Skor Max
            'F' => 15,  // Jenis
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return 'Instrumen Penilaian';
    }
}
